#include "global.h"
#include <errno.h>
#include "NetworkSyncManager.h"
#include "arch/LoadingWindow/LoadingWindow.h"
#include "ezsockets.h"
#include "ScreenManager.h"
#include "CryptManager.h"
#include "LocalizedString.h"
#include "RageLog.h"
#include "RageTimer.h"
#include "ProductInfo.h"
#include "LuaManager.h"

// todo: localize all the strings.

NetworkSyncManager *NSMAN;
const float fWaitTimeout = 5.0f;

NetworkSyncManager::NetworkSyncManager( LoadingWindow *ld )
{
	ld->SetText( "init sm5te client network" );

	// init player socket
	m_PlayerSocket = new EzSockets;
	m_PlayerSocket->blocking = false;

	m_NetConnected = false;
	m_OnlineServer = false;
	m_ServerProtocolVersion = 0;
	m_StartupStatus = StartupStatus_NotTried;

	CheckCommandLine();
}

NetworkSyncManager::~NetworkSyncManager()
{
	if(m_NetConnected)
		m_PlayerSocket->close();
	SAFE_DELETE( m_PlayerSocket );
}

// [NetProtocol] generic packet sending function
void NetworkSyncManager::SendPacket(NetworkPacket *p)
{
	m_PlayerSocket->SendPack( (char*)p->Data , p->Position );
}

void NetworkSyncManager::CheckCommandLine()
{
	RString serverAddr;

	if( GetCommandlineArgument( "netip", &serverAddr ) )
	{
		if( TryConnection(serverAddr) )
			OnConnect();
	}
}

bool NetworkSyncManager::TryConnection( const RString& serverAddr )
{
	RString sAddress;
	unsigned short iPort;
	size_t cLoc = serverAddr.find( ':' );
	if( serverAddr.find( ':' ) != RString::npos )
	{
		sAddress = serverAddr.substr( 0, cLoc );
		char* cEnd;
		errno = 0;
		iPort = (unsigned short)strtol( serverAddr.substr( cLoc + 1 ).c_str(), &cEnd, 10 );
		if( *cEnd != 0 || errno != 0 )
		{
			m_StartupStatus = StartupStatus_Failed;
			LOG->Warn( "TryConnection failed: Invalid port" );
			return false;
		}
	}
	else
	{
		sAddress = serverAddr;
		iPort = DefaultPort;
	}

	CloseConnection();
	if( !Connect(sAddress.c_str(), iPort) )
	{
		m_StartupStatus = StartupStatus_Failed;
		LOG->Warn( "Network Sync Manager failed to connect" );
		return false;
	}

	m_StartupStatus = StartupStatus_Success;
	return true;
}

void NetworkSyncManager::CloseConnection()
{
	if(!m_NetConnected)
		return;

	// reset things
	m_NetConnected = false;
	m_OnlineServer = false;
	m_ServerProtocolVersion = 0;
	m_StartupStatus = StartupStatus_NotTried;
	m_PlayerSocket->close();
}

bool NetworkSyncManager::Connect( const RString& addr, unsigned short port )
{
	m_PlayerSocket->create();
	m_NetConnected = m_PlayerSocket->connect(addr,port);
	return m_NetConnected;
}

void NetworkSyncManager::OnConnect()
{
	m_NetConnected = true;

	if(!GetServerStatus())
	{
		// can't join server
		return;
	}
	if(!SendHello())
	{
		// hello was not able to be sent.
		return;
	}
}

bool NetworkSyncManager::GetServerStatus()
{
	// compose status packet
	m_Packet.Clear();
	m_Packet.Write1(ClientCommand::ServerStatus);
	m_Packet.Write1(ProtocolVersion);
	m_Packet.WriteString(RString(PRODUCT_ID));
	m_Packet.WriteString(RString(PRODUCT_VER));

	// This packet must get through, so we block for it.
	m_PlayerSocket->blocking = true;
	SendPacket(&m_Packet);

	// wait for ten seconds or if there's an error.
	RageTimer waitTime;
	bool breakLoop = false;
	uint8_t serverResponse = 0;
	while(waitTime.GetDeltaTime() < fWaitTimeout && !breakLoop)
	{
		// wait for a response
		m_Packet.Clear();
		if( m_PlayerSocket->ReadPack((char *)&m_Packet,MAX_PACKET_BUFFER_SIZE) < 1 )
		{
			breakLoop = true; // break if there's a socket problem.
			serverResponse = 2;
		}
		if( m_Packet.Read1() == ServerCommand::ServerStatus )
		{
			breakLoop = true;
			serverResponse = 1;
		}
	}
	m_PlayerSocket->blocking = false;

	if(serverResponse == 1)
	{
		RString serverStatus;
		uint8_t serverType;
		int playersConnected, maxPlayers;

		// read the rest of the packet:
		m_ServerVersion = m_Packet.ReadString(); // server version
		m_ServerProtocolVersion = m_Packet.Read1(); // protocol version
		m_ServerName = m_Packet.ReadString(); // server name
		serverStatus = m_Packet.ReadString(); // server status
		serverType = m_Packet.Read1(); // server type
		playersConnected = m_Packet.Read4(); // num players connected
		maxPlayers = m_Packet.Read4(); // max players connected

		// check if this is an online server. if not, it's a LAN server.
		m_OnlineServer = (serverType == 0);

		// assess server status
		if(serverStatus == "Online")
		{
			return true;
		}
		else if(serverStatus == "PlayerLimitReached")
		{
			// too many players
			LOG->Warn("Server maximum of %i players already reached.",maxPlayers);
			return false;
		}
		else if(serverStatus == "OutdatedProtocolC")
		{
			// client is too old
			LOG->Warn("Client protocol version (%i) < Server protocol version (%i)",ProtocolVersion,m_ServerProtocolVersion);
			return false;
		}
		else if(serverStatus == "OutdatedProtocolS")
		{
			// Server is too old
			LOG->Warn("Client protocol version (%i) > Server protocol version (%i)",ProtocolVersion,m_ServerProtocolVersion);
			return false;
		}
		else if(serverStatus == "Maintenence")
		{
			// server is down for maintenence
			return false;
		}
	}
	else if(serverResponse == 0)
	{
		// no connection made
		LOG->Trace("[NetworkSyncManager] No connection could be made.");
		return false;
	}
	else if(serverResponse == 2)
	{
		// socket problem
		LOG->Trace("[NetworkSyncManager] Problem with the socket.");
		return false;
	}
	else
	{
		// unknown error
		LOG->Trace("[NetworkSyncManager] unknown error.");
		return false;
	}
	LOG->Trace("[NetworkSyncManager] what in the hell? we shouldn't be here");
	return false;
}

bool NetworkSyncManager::SendHello()
{
	m_Packet.Clear();
	m_Packet.Write1(ClientCommand::Hello);
	m_Packet.Write1(ProtocolVersion);
	m_Packet.WriteString(RString(PRODUCT_ID));
	m_Packet.WriteString(RString(PRODUCT_VER));

	// This packet must get through, so we block for it.
	m_PlayerSocket->blocking = true;
	SendPacket(&m_Packet);

	// reduced wait time
	RageTimer waitTime;
	bool breakLoop = false;
	while(waitTime.GetDeltaTime() < fWaitTimeout && !breakLoop)
	{
		// wait for a response
		m_Packet.Clear();
		if( m_PlayerSocket->ReadPack((char *)&m_Packet,MAX_PACKET_BUFFER_SIZE) < 1 )
		{
			return false;
		}
		if( m_Packet.Read1() == ServerCommand::Hello )
		{
			breakLoop = true;
		}
	}
	m_PlayerSocket->blocking = false;

	// read the rest of the hello packet
	uint8_t serverType;
	m_ServerVersion = m_Packet.ReadString(); // server version
	m_ServerProtocolVersion = m_Packet.Read1(); // protocol version
	m_ServerName = m_Packet.ReadString(); // server name
	serverType = m_Packet.Read1(); // server type

	// check if this is an online server. if not, it's a LAN server.
	m_OnlineServer = (serverType == 0);

	return true;
}

bool NetworkSyncManager::SendLogin(int iPlayerSide, RString sUsername, RString sPlainPassword)
{
	m_Packet.Clear();
	m_Packet.Write1(ClientCommand::Login);
	// xxx: everything after this is out of date! -freem

	m_Packet.Write1(iPlayerSide);			// player side
	m_Packet.WriteString(sUsername);
	m_Packet.WriteString(sPlainPassword);		// XXX: DONT FUCKING KEEP THIS AROUND FOR LONG!!!!!! YOU'LL GET RAPED BY ALL THE HACKERS

	// This packet must get through, so we block for it.
	m_PlayerSocket->blocking = true;
	SendPacket(&m_Packet);

	RageTimer waitTime;
	bool breakLoop = false;
	while(waitTime.GetDeltaTime() < fWaitTimeout && !breakLoop)
	{
		// wait for a response
		m_Packet.Clear();
		if( m_PlayerSocket->ReadPack((char *)&m_Packet,MAX_PACKET_BUFFER_SIZE) < 1 )
		{
			return false;
		}
		if( m_Packet.Read1() == ServerCommand::Login )
		{
			breakLoop = true;
		}
	}
	m_PlayerSocket->blocking = false;

	// read the rest of the Login reply packet

	return true;
}

RString NetworkSyncManager::GetServerName()		{ return m_ServerName; }
int NetworkSyncManager::GetSMOnlineSalt()		{ return m_iSalt; }

void NetworkSyncManager::DisplayStartupStatus()
{
	RString sMessage("");
	switch(m_StartupStatus)
	{
		case 1:
			sMessage = ssprintf("Connection to %s successful",m_ServerName.c_str());
			break;
		case 2:
			sMessage = "Connection Failed";
			break;
	}
	SCREENMAN->SystemMessage( sMessage );
}

void NetworkSyncManager::Update( float fDeltaTime )
{
	if(m_NetConnected)
		ProcessInput();
}

void NetworkSyncManager::ProcessInput()
{
	if ((m_PlayerSocket->state != m_PlayerSocket->skCONNECTED) || m_PlayerSocket->IsError())
	{
		// connection dropped
		SCREENMAN->SystemMessageNoAnimate( "Connection dropped!" );
		m_NetConnected = false;
		m_OnlineServer = false;
		m_PlayerSocket->close();
		return;
	}

	// load new data
	m_PlayerSocket->update();
	m_Packet.Clear();

	int packetSize;
	while( (packetSize = m_PlayerSocket->ReadPack((char *)&m_Packet,MAX_PACKET_BUFFER_SIZE)) > 0 )
	{
		m_Packet.Size = packetSize;
		int command = m_Packet.Read1();
		// check if command is from server.

		HandleCommand(command);
		m_Packet.Clear();
	}
}

void NetworkSyncManager::HandleCommand(int command)
{
	LOG->Trace("[NetworkSyncManager::HandleCommand] Command %i",command);
	/*
	switch(command)
	{
		
	}
	*/
}

RString NetworkSyncManager::MD5Hex( const RString &sInput ) 
{
	return BinaryToHex( CryptManager::GetMD5ForString(sInput) ).MakeUpper();
}

// lua time
static bool CloseNetworkConnection() { NSMAN->CloseConnection(); return true; }

LuaFunction( IsNetConnected,	NSMAN->m_NetConnected )
LuaFunction( IsNetSMOnline,		NSMAN->m_OnlineServer )
LuaFunction( GetServerName,		NSMAN->GetServerName() )
LuaFunction( CloseConnection,	CloseNetworkConnection() )

// old stuff
void NetworkSyncManager::ReportNSSOnOff( int i ) { }
void NetworkSyncManager::ReportScore( int playerID, int step, int score, int combo, float offset ) { }
void NetworkSyncManager::ReportSongOver() { }
void NetworkSyncManager::ReportStyle() {}
void NetworkSyncManager::StartRequest( short position ) { }
void NetworkSyncManager::SendChat( const RString& message ) { }
void NetworkSyncManager::SelectUserSong() { }

/*
 * (c) 2012 AJ Kelly
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons to
 * whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF
 * THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS
 * INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT
 * OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
