#include "global.h"
#include "ScreenServerConnect.h"
#include "NetworkSyncManager.h"
#include "ScreenSelectMaster.h"
#include "GameState.h"
#include "ScreenManager.h"
#include "InputEventPlus.h"
#include "GameCommand.h"
#include "RageLog.h"
#include "ScreenTextEntry.h"
#include "SpecialFiles.h"
#include "JsonUtil.h"

#define CHOICE( s )		THEME->GetMetric (m_sName,ssprintf("Choice%s",s.c_str()))

AutoScreenMessage( SM_DoneConnecting );
AutoScreenMessage( SM_AddingServerName );	// after entering the server name
AutoScreenMessage( SM_DoneAddingServer );	// after entering the server address

Preference<RString> g_sLastServer( "LastConnectedServer",	"" );

REGISTER_SCREEN_CLASS( ScreenServerConnect );

// serializes internal structures to json data
static void Serialize( const ServerEntry &e, Json::Value &root )
{
	root["Name"] = e.name;
	root["Address"] = e.address;
}

// de-serializes json data into internal structures.
static void Deserialize( ServerEntry &e, const Json::Value &root )
{
	e.name = root["Name"].asString();
	e.address = root["Address"].asString();
}

ScreenServerConnect::ScreenServerConnect()
{
	m_iChoice = 0;
	m_sNewServerName = "";
	m_sNewServerAddress = "";
	m_jServerList = Json::Value(Json::arrayValue);
}

void ScreenServerConnect::Init()
{
	WRAP_SCROLLER.Load( m_sName, "WrapScroller" );
	LOOP_SCROLLER.Load( m_sName, "LoopScroller" );
	ALLOW_REPEATING_INPUT.Load( m_sName, "AllowRepeatingInput" );
	SCROLLER_SECONDS_PER_ITEM.Load( m_sName, "ScrollerSecondsPerItem" );
	SCROLLER_NUM_ITEMS_TO_DRAW.Load( m_sName, "ScrollerNumItemsToDraw" );
	SCROLLER_TRANSFORM.Load( m_sName, "ScrollerTransform" );
	SCROLLER_SUBDIVISIONS.Load( m_sName, "ScrollerSubdivisions" );

	ScreenWithMenuElements::Init();

	// Load choices
	{
		// ideally, we want to LoadServersFromJson() and have the scroller
		// contents be: quick connect, {server list}, add server, exit.

		// todo: localize the strings
		GameCommand qc;
		qc.ApplyCommitsScreens( false );
		qc.m_sName = "QuickConnect";
		Commands cmd1 = ParseCommands( CHOICE( RString("QuickConnect") ) );
		qc.Load( 0, cmd1 );
		m_aGameCommands.push_back(qc);

		GameCommand ns;
		ns.ApplyCommitsScreens( false );
		ns.m_sName = "NewServer";
		Commands cmd2 = ParseCommands( CHOICE( RString("NewServer") ) );
		ns.Load( 1, cmd2 );
		m_aGameCommands.push_back(ns);

		if( !LoadServersFromJson() )
		{
			LOG->Trace("[ScreenServerConnect] Unable to read Servers.json; attempting to create new file...");
			if( !SaveServerList() )
			{
				// todo: be specific on why it can't save
				LOG->Trace("[ScreenServerConnect] Unable to save Servers.json.");
			}
			else
			{
				LOG->Trace("[ScreenServerConnect] Successfully created Servers.json");
			}
		}
		else
		{
			if( m_jServerList.size() > 0 && m_ServerEntries.size() > 0 )
			{
				// time to look through m_ServerEntries
				for(unsigned int i=0; i < m_ServerEntries.size(); i++)
				{
					GameCommand gc;
					gc.ApplyCommitsScreens( false );
					gc.m_sName = ssprintf("Server%u",i);
					gc.m_sText = m_ServerEntries[i].name;
					Commands cmd;
					gc.Load( 2+i, cmd );
					m_aGameCommands.push_back(gc);
				}
			}
		}

		GameCommand exit;
		exit.ApplyCommitsScreens( true );
		exit.m_sName = "Exit";
		Commands cmd3 = ParseCommands( CHOICE( RString("Exit") ) );
		exit.Load( m_aGameCommands.size(), cmd3 );
		m_aGameCommands.push_back(exit);
	}

	m_vScrollList.resize(m_aGameCommands.size());
	for( unsigned c=0; c < m_aGameCommands.size(); c++ )
	{
		GameCommand& gc = m_aGameCommands[c];
		LuaThreadVariable var( "GameCommand", LuaReference::Create(&gc) );

		vector<RString> vs;
		vs.push_back( "Scroll" );
		RString sElement = join( " ", vs );
		m_vScrollList[c].Load( THEME->GetPathG(m_sName,sElement) );
		RString sName = "Scroll" "Choice" + gc.m_sName;
		m_vScrollList[c]->SetName(sName);
		m_ServerScroller.AddChild(m_vScrollList[c]);
	}

	m_ServerScroller.SetLoop( LOOP_SCROLLER );
	m_ServerScroller.SetNumItemsToDraw( SCROLLER_NUM_ITEMS_TO_DRAW );
	m_ServerScroller.Load2();
	m_ServerScroller.SetTransformFromReference( SCROLLER_TRANSFORM );
	m_ServerScroller.SetSecondsPerItem( SCROLLER_SECONDS_PER_ITEM );
	m_ServerScroller.SetNumSubdivisions( SCROLLER_SUBDIVISIONS );
	m_ServerScroller.SetName( "Scroller" );
	LOAD_ALL_COMMANDS_AND_SET_XY( m_ServerScroller );
	this->AddChild( &m_ServerScroller );
}

bool ScreenServerConnect::LoadServersFromJson()
{
	if( !JsonUtil::LoadFromFileShowErrors(m_jServerList, SpecialFiles::SERVERS_JSON_PATH) )
		return false;

	for( unsigned i=0; i<m_jServerList.size(); i++ )
	{
		Json::Value entry = m_jServerList[i];
		ServerEntry server;
		Deserialize(server, entry);
		m_ServerEntries.push_back(server);
	}

	return true;
}

bool ScreenServerConnect::AddServerToList(RString sName, RString sAddr)
{
	// no blank values thanks
	if( sName.empty() ) return false;
	if( sAddr.empty() ) return false;

	const ServerEntry entry = ServerEntry(sName,sAddr);
	m_ServerEntries.push_back(entry);

	Json::Value newServer;
	Serialize(entry, newServer);
	m_jServerList.append(newServer);

	return true;
}

bool ScreenServerConnect::SaveServerList()
{
	return JsonUtil::WriteFile( m_jServerList, SpecialFiles::SERVERS_JSON_PATH, false );
}

void ScreenServerConnect::BeginScreen()
{
	ScreenWithMenuElements::BeginScreen();
}

void ScreenServerConnect::HandleScreenMessage( const ScreenMessage SM )
{
	if( SM == SM_DoneConnecting )
	{
		if( !ScreenTextEntry::s_bCancelledLast )
		{
			RString newAddr = ScreenTextEntry::s_sLastAnswer;

			if(NSMAN->TryConnection(newAddr))
			{
				NSMAN->OnConnect();
			}

			NSMAN->DisplayStartupStatus();
			g_sLastServer.Set( newAddr );
		}
	}
	else if( SM == SM_AddingServerName )
	{
		if( !ScreenTextEntry::s_bCancelledLast )
		{
			// finished setting a new server's name.
			m_sNewServerName = ScreenTextEntry::s_sLastAnswer;
			ScreenTextEntry::TextEntry( SM_DoneAddingServer, "Enter Server Address", "", 256 );
		}
		else
		{
			ResetNewServerValues();
		}
	}
	else if( SM == SM_DoneAddingServer )
	{
		if( !ScreenTextEntry::s_bCancelledLast )
		{
			// finished adding a server to the list (name & address)
			m_sNewServerAddress = ScreenTextEntry::s_sLastAnswer;
			if( AddServerToList(m_sNewServerName, m_sNewServerAddress) )
			{
				SCREENMAN->SystemMessage( ssprintf("%s | %s", m_sNewServerName.c_str(), m_sNewServerAddress.c_str()) );
				// save servers.json, probably.
				SaveServerList();
			}
			// then reset values for the next one.
			ResetNewServerValues();
		}
		else
		{
			ResetNewServerValues();
		}
	}
	else if( SM == SM_BeginFadingOut )	// Screen is starting to tween out.
	{
		const GameCommand &gc = m_aGameCommands[m_iChoice];

		if(!gc.m_bInvalid)
			gc.Apply(GAMESTATE->GetMasterPlayerNumber());

		StopTimer();
		ASSERT( !IsTransitioning() );
		StartTransitioningScreen( SM_GoToNextScreen );
	}
	ScreenWithMenuElements::HandleScreenMessage( SM );
}

bool ScreenServerConnect::Move( PlayerNumber pn, MenuDir dir )
{
	int iSwitchToIndex = m_iChoice;
	if(dir == MenuDir_Up)
	{
		iSwitchToIndex -= 1;
	}
	else if(dir == MenuDir_Down)
	{
		iSwitchToIndex += 1;
	}

	if(iSwitchToIndex < 0 || iSwitchToIndex >= (int)m_aGameCommands.size())
		return false;

	return ChangeSelection( pn, dir, iSwitchToIndex );
}

void ScreenServerConnect::MenuUp( const InputEventPlus &input )
{
	PlayerNumber pn = input.pn;
	if( m_fLockInputSecs > 0)
		return;
	if( input.type == IET_RELEASE )
		return;
	if( input.type != IET_FIRST_PRESS )
	{
		if( !ALLOW_REPEATING_INPUT )
			return;
		if( m_TrackingRepeatingInput != input.MenuI )
			return;
	}
	if( Move(pn, MenuDir_Up) )
	{
		m_TrackingRepeatingInput = input.MenuI;
		//m_soundChange.Play();
		MESSAGEMAN->Broadcast( (MessageID)(Message_MenuUpP1+pn) );
		MESSAGEMAN->Broadcast( (MessageID)(Message_MenuSelectionChanged) );
	}
}

void ScreenServerConnect::MenuDown( const InputEventPlus &input )
{
	PlayerNumber pn = input.pn;
	if( m_fLockInputSecs > 0)
		return;
	if( input.type == IET_RELEASE )
		return;
	if( input.type != IET_FIRST_PRESS )
	{
		if( !ALLOW_REPEATING_INPUT )
			return;
		if( m_TrackingRepeatingInput != input.MenuI )
			return;
	}
	if( Move(pn, MenuDir_Down) )
	{
		m_TrackingRepeatingInput = input.MenuI;
		//m_soundChange.Play();
		MESSAGEMAN->Broadcast( (MessageID)(Message_MenuDownP1+pn) );
		MESSAGEMAN->Broadcast( (MessageID)(Message_MenuSelectionChanged) );
	}
}

bool ScreenServerConnect::ChangeSelection( PlayerNumber pn, MenuDir dir, int iNewChoice )
{
	if( iNewChoice == m_iChoice )
		return false; // already there

	// old and new
	const int iOldChoice = m_iChoice;
	m_iChoice = iNewChoice;

	ActorScroller &scroller = m_ServerScroller;
	vector<AutoActor> &vScroll = m_vScrollList;
	{
		// HACK: We can't tell from the option orders whether or not we wrapped.
		// For now, assume that the order is increasing up to down.
		int iPressedDir = (dir == MenuDir_Up) ? -1 : +1;
		int iActualDir = (iOldChoice < iNewChoice) ? +1 : -1;

		if( iPressedDir != iActualDir )	// wrapped
		{
			//float fItem = scroller.GetCurrentItem();
			/*
			int iNumChoices = m_aGameCommands.size();
			fItem += iActualDir * iNumChoices;
			scroller.SetCurrentAndDestinationItem( fItem );
			*/
		}
	}
	scroller.SetDestinationItem( (float)iNewChoice );

	vScroll[iOldChoice]->PlayCommand( "LoseFocus" );
	vScroll[iNewChoice]->PlayCommand( "GainFocus" );

	return true;
}

void ScreenServerConnect::MenuStart( const InputEventPlus &input )
{
	if( input.type != IET_FIRST_PRESS )
		return;
	if( m_fLockInputSecs > 0 )
		return;

	PlayerNumber pn = input.pn;
	const GameCommand &mc = m_aGameCommands[m_iChoice];
	if( mc.m_sScreen.empty() )
	{
		Message msg( MessageIDToString((MessageID)(Message_MenuStartP1+pn)) );
		msg.SetParam( "Index", m_iChoice );
		MESSAGEMAN->Broadcast( msg );
	}

	if(m_iChoice == 0)
	{
		// quick connect
		ScreenTextEntry::TextEntry( SM_DoneConnecting, "Enter Network Address", g_sLastServer, 256 );
	}
	else if(m_iChoice == 1)
	{
		// add server, which is a two-part process. (currently implemented using
		// 2x ScreenTextEntry and the posted ScreenMessages)
		ScreenTextEntry::TextEntry( SM_AddingServerName, "Enter Server Name", "", 256 );
	}
	else if(m_iChoice == (int)m_aGameCommands.size()-1)
	{
		// exit
		m_sNextScreen = mc.m_sScreen;
		this->PostScreenMessage( SM_BeginFadingOut, 0 );
		return;
	}
	else
	{
		// a server from the list. we want to connect to it.
		unsigned int serverIndex = m_iChoice-2;
		if( NSMAN->TryConnection( m_ServerEntries[serverIndex].address ) )
		{
			NSMAN->OnConnect();
		}

		NSMAN->DisplayStartupStatus();
	}
}

void ScreenServerConnect::MenuSelect( const InputEventPlus &input )
{
	if( m_iChoice >= 2 && m_iChoice < (int)m_aGameCommands.size()-1 )
	{
		// show a MiniMenu if m_iChoice is a server.
		SCREENMAN->SystemMessage( "Need to implement select button stuff aaa" );
	}
}

void ScreenServerConnect::TweenOnScreen()
{
	for( unsigned c=0; c<m_aGameCommands.size(); c++ )
	{
		m_vScrollList[c]->PlayCommand( int(c) == m_iChoice? "GainFocus":"LoseFocus" );
		m_vScrollList[c]->FinishTweening();
	}

	m_ServerScroller.SetCurrentAndDestinationItem( (float)m_iChoice );

	ScreenWithMenuElements::TweenOnScreen();
}

void ScreenServerConnect::TweenOffScreen()
{
	ScreenWithMenuElements::TweenOffScreen();

	for( unsigned c=0; c<m_aGameCommands.size(); c++ )
	{
		bool selectedByPlayer = (m_iChoice == (int)c);
		m_vScrollList[c]->PlayCommand( selectedByPlayer? "OffFocused":"OffUnfocused" );
	}
}

int ScreenServerConnect::GetSelectionIndex( PlayerNumber pn ){ return m_iChoice; }

void ScreenServerConnect::UpdateSelectableChoices()
{
	// important; needs to run when adding/deleting a server, or connecting (so
	// we can show the disconnect option)
}

void ScreenServerConnect::MenuBack( const InputEventPlus &input )
{
	Cancel( SM_GoToPrevScreen );
}

//LUA_REGISTER_DERIVED_CLASS( ScreenServerConnect, ScreenWithMenuElements )

/*
 * (c) 2012 AJ Kelly
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons to
 * whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF
 * THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS
 * INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT
 * OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
