#include "global.h"
#include "TextBox.h"
#include "ActorUtil.h"
#include "FontCharAliases.h"
#include "InputEventPlus.h"
#include "RageInput.h"

REGISTER_ACTOR_CLASS( TextBox );

/* todo:
 * handle focus changes
 * allow themer to change password character from '*' to m_sFilterChar
 */

TextBox::TextBox()
{
	m_bHasInputFocus = false;
	m_sAnswer = RStringToWstring("");
	m_sPrevAnswer = RStringToWstring("");
}

void TextBox::Load( const RString &sMetricsGroup )
{
	m_sMetricsGroup = sMetricsGroup;

	m_bPassword.Load(m_sMetricsGroup,"Password");
	m_sFilterChar.Load(m_sMetricsGroup,"FilterCharacter");

	m_Background.Load( THEME->GetPathG(m_sMetricsGroup,"Background") );
	m_Background->SetName( "Background" );
	ActorUtil::LoadAllCommandsAndSetXYAndOnCommand( m_Background, m_sMetricsGroup );
	this->AddChild( m_Background );

	m_Label.Load( THEME->GetPathG(m_sMetricsGroup,"Label") );
	m_Label->SetName( "Label" );
	ActorUtil::LoadAllCommandsAndSetXYAndOnCommand( m_Label, m_sMetricsGroup );
	this->AddChild( m_Label );

	m_InputText.SetName( "Text" );
	m_InputText.LoadFromFont( THEME->GetPathF(m_sMetricsGroup,"Text") );
	ActorUtil::LoadAllCommandsAndSetXYAndOnCommand( m_InputText, m_sMetricsGroup );
	this->AddChild( &m_InputText );
}

void TextBox::GainFocus() { this->PlayCommand("GainFocus"); }
void TextBox::LoseFocus() { this->PlayCommand("LoseFocus"); }

void TextBox::SetInputFocus(bool bFocus)
{
	m_bHasInputFocus = bFocus;
	this->PlayCommand(bFocus ? "GainInputFocus" : "LoseInputFocus");
}

bool TextBox::Input( const InputEventPlus &input )
{
	if( input.DeviceI == DeviceInput(DEVICE_KEYBOARD, KEY_BACK) )
	{
		switch( input.type )
		{
			case IET_FIRST_PRESS:
			case IET_REPEAT:
				Backspace();
				return true;
			default:
				break;
		}
	}
	// todo: allow keypad enter?
	else if( input.DeviceI == DeviceInput(DEVICE_KEYBOARD, KEY_ENTER))
	{
		if( input.type == IET_FIRST_PRESS )
			End(false);
		return true;
	}
	else if( input.DeviceI == DeviceInput(DEVICE_KEYBOARD, KEY_ESC))
	{
		// cancel answer
		if( input.type == IET_FIRST_PRESS )
			End(true);
		return true;
	}
	else if( input.type == IET_FIRST_PRESS )
	{
		wchar_t c = INPUTMAN->DeviceInputToChar(input.DeviceI,true);
		if( c >= L' ' ) 
		{
			TryAppendToAnswer( WStringToRString(wstring()+c) );
		}
		return true;
	}

	return false; // did not handle input
}

// upon ending input via enter or escape.
void TextBox::End(bool bCanceled)
{
	SetInputFocus(false);
	if(bCanceled)
	{
		m_sAnswer = m_sPrevAnswer;
	}
	else
	{
		m_sPrevAnswer = m_sAnswer;
	}
}

void TextBox::UpdateAnswerText()
{
	RString s;

	// todo: get m_sFilterChar working
	if( m_bPassword )
		s = RString( m_sAnswer.size(), '*' );
	else
		s = WStringToRString(m_sAnswer);

	FontCharAliases::ReplaceMarkers( s );
	m_InputText.SetText( s );
}

RString TextBox::GetAnswerSafe()
{
	return m_bPassword ? RString( m_sAnswer.size(), '*' ) : WStringToRString(m_sAnswer);
}

void TextBox::TryAppendToAnswer( RString s )
{
	wstring sNewAnswer = m_sAnswer+RStringToWstring(s);
	m_sAnswer = sNewAnswer;
	UpdateAnswerText();
}

void TextBox::Backspace()
{
	if( m_sAnswer.empty() )
	{
		return;
	}
	m_sAnswer.erase( m_sAnswer.end()-1 );
	UpdateAnswerText();
}

// lua start
#include "LuaBinding.h"
/** @brief Allow Lua to have access to the TextBox. */ 
class LunaTextBox : public Luna<TextBox>
{
public:
	static int GetAnswer( T* p, lua_State *L ) { lua_pushstring( L, p->GetAnswerSafe() ); return 1; }
	DEFINE_METHOD( HasInputFocus, GetInputFocus() )

	LunaTextBox()
	{
		ADD_METHOD( GetAnswer );
		ADD_METHOD( HasInputFocus );
	}
};

LUA_REGISTER_DERIVED_CLASS( TextBox, ActorFrame )

/*
 * (c) 2012 AJ Kelly
 * All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons to
 * whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF
 * THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR HOLDERS
 * INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL INDIRECT
 * OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */
