local gs = {}
local path = '/Save/GrooveStats/'

local timeout = 5

-- Reduce the chart to it's smallest unique representable form.
local MinimizeChart = function(chartString)
	local function MinimizeMeasure(measure)
		local minimal = false
		-- We can potentially minimize the chart to get the most compressed
		-- form of the actual chart data.
		-- NOTE(teejusb): This can be more compressed than the data actually
		-- generated by StepMania. This is okay because the charts would still
		-- be considered equivalent.
		-- E.g. 0000                      0000
		--      0000  -- minimized to -->
		--      0000
		--      0000
		--      StepMania will always generate the former since quarter notes are
		--      the smallest quantization.
		while not minimal and #measure % 2 == 0 do
			-- If every other line is all 0s, we can minimize the measure.
			local allZeroes = true
			for i=2, #measure, 2 do
				-- Check if the row is NOT all zeroes (thus we can't minimize).
				if measure[i] ~= string.rep('0', measure[i]:len()) then
					allZeroes = false
					break
				end
			end

			if allZeroes then
				-- To remove every other element while keeping the
				-- indices valid, we iterate from [2, len(t)/2 + 1].
				-- See the example below (where len(t) == 6).

				-- index: 1 2 3 4 5 6  -> remove index 2
				-- value: a b a b a b

				-- index: 1 2 3 4 5    -> remove index 3
				-- value: a a b a b

				-- index: 1 2 3 4      -> remove index 4
				-- value: a a a b

				-- index: 1 2 3
				-- value: a a a
				for i = 2, #measure * 0.5 + 1 do
					table.remove(measure, i)
				end
			else
				minimal = true
			end
		end
	end

	local finalChartData = {}
	local curMeasure = {}
	for line in chartString:gmatch('[^\n]+') do
		-- If we hit a comma, that denotes the end of a measure.
		-- Try to minimize it, and then add it to the final chart data with
		-- the delimiter.
		-- Note: The ending semi-colon has been stripped out.
		if line == ',' then
			MinimizeMeasure(curMeasure)

			for row in ivalues(curMeasure) do
				table.insert(finalChartData, row)
			end
			table.insert(finalChartData, ',')
			-- Just keep removing the first element to clear the table.
			-- This way we don't need to wait for the GC to cleanup the unused values.
			--[[
			for i=1, #curMeasure do
				table.remove(curMeasure, 1)
			end
			--]]
			curMeasure = {}
		else
			table.insert(curMeasure, line)
		end
	end

	-- Add the final measure.
	if #curMeasure > 0 then
		MinimizeMeasure(curMeasure)

		for row in ivalues(curMeasure) do
			table.insert(finalChartData, row)
		end
	end

	return table.concat(finalChartData, '\n')
end

-- NoteData: {10, 1, 'TapNoteType_Mine', 'TapNote_4th', length = nil}
-- TimeSignature: {10, 4, 4}
local function NoteDataToChartString(nd, ts)
	local minichart = {}
	local start_i, cur_i = 1, 1
	local line_len = GAMESTATE:GetCurrentStyle():ColumnsPerPlayer()
	local allZero = true
	while charttable[cur_i] ~= ';' do
		local line = charttable[cur_i]
		if line ~= string.rep('0', line_len) then
			allZero = false
		end
		if line == ',' then
		end
	end
end

local function minify(charttable)
	local function minifymeasure(m)
		local finished = false
		while not finished and #m % 2 == 0 do
			local allzero = true
			for i = 2, #m, 2 do
				if m[i] ~= string.rep('0', m[i]:len()) then
					allzero = false
					break
				end
			end
			if allzero then
				for i = 2, #m * 0.5 + 1 do
					table.remove(m, i)
				end
			else
				finished = true
			end
		end
	end
	local curMeasure, finalData = {}, {}
	for line in ivalues(charttable) do
		if line == ',' then
			minifymeasure(curMeasure)
			for row in ivalues(curMeasure) do
				table.insert(finalData, row)
			end
			table.insert(finalData, ',')
			curMeasure = {}
		else
			table.insert(curMeasure, line)
		end
	end
	if #curMeasure > 0 then
		minifymeasure(curMeasure)
		for row in ivalues(curMeasure) do
			table.insert(finalData, row)
		end
	end
	return table.concat(finalData, '\n')
end

local NormalizeFloatDigits = function(param)
	local function NormalizeDecimal(decimal)
		-- Remove any control characters from the string to prevent conversion failures.
		decimal = decimal:gsub("%c", "")
		local rounded = tonumber(decimal)

		-- Round to 3 decimal places
		local mult = 10^3
		rounded = (rounded * mult + 0.5 - (rounded * mult + 0.5) % 1) / mult
		return string.format("%.3f", rounded)
	end

	local paramParts = {}
	for beat_bpm in param:gmatch('[^,]+') do
		local beat, bpm = beat_bpm:match('(.+)=(.+)')
		table.insert(paramParts, NormalizeDecimal(beat) .. '=' .. NormalizeDecimal(bpm))
	end
	return table.concat(paramParts, ',')
end

local function CharReplace(str, pos, char)
	return str:sub(1, pos - 1)..char..str:sub(pos + 1)
end

local function GSPrefs(plr)
	local ret = {}
	local pn = PlayerNumber:Reverse()[plr] + 1
	local path = PROFILEMAN:GetProfileDir(ProfileSlot[pn])
	if not path then return ret end
	local data = IniFile.ReadFile(path..'GrooveStats.ini')
	if not data or not data.GrooveStats then return ret end
	return data.GrooveStats
end

function gs.Enabled()
	if not ProductVersion():find('%.4') then return false end
	return (#FILEMAN:GetDirListing(path, true, false) > 0)
end

function gs.GetAPI(plr)
	local gsData = GSPrefs(plr)
	return gsData.ApiKey
end

function gs.IsPadPlayer(plr)
	local ret = false
	local gsData = GSPrefs(plr)
	if gsData.IsPadPlayer then
		ret = (tonumber(gsData.IsPadPlayer) == 1 and true) or false
	end
	return ret
end

function gs.timeout(s)
	if not s then return timeout end
	timeout = s
end

function gs.ChartHash(plr)
	local song = GAMESTATE:GetCurrentSong()
	local filepath = song:GetSongFilePath()
	local filetype = filepath:sub(1 - filepath:reverse():find('%.'))
	local chart = GAMESTATE:GetCurrentSteps(plr)
	local bpmtable = chart:GetTimingData():GetBPMsAndTimes()
	local bpms = NormalizeFloatDigits(table.concat(bpmtable, ','))
	local stepstype = ToEnumShortString(chart:GetStepsType()):lower()
	local diff = ToEnumShortString(chart:GetDifficulty()):lower()
	local desc = chart:GetDescription()
	local data = File.Read(filepath)
	local nd
	for i, v in ipairs(song:GetAllSteps()) do
		if v == chart then
			nd = song:GetNoteData(i)
		end
	end
	
	-- i give up. ~Sudo
	--[[
	local ndtable = {}
	local last_beat = math.ceil(nd[#nd][1] + (nd[#nd].length or 0)) + 1
	local last_line = last_beat * 64
	local line_len = GAMESTATE:GetCurrentStyle():ColumnsPerPlayer()
	-- First, pad our notedata table with all zeroes.
	for i = 1, last_line do
		table.insert(ndtable, string.rep('0', line_len))
	end
	-- Next, fill in with our notedata.

	for v in ivalues(nd) do
		local beat = math.ceil(v[1] * 64) + 1 -- align with our index in ndtable
		local col = v[2]
		local type = v[3]
		local len = math.ceil((v.length or 0) * 64)
		if type:find('Sub') then
			if type == TapNoteSubType[1] then -- 'TapNoteSubType_Hold'
				type = '2'
			elseif type == TapNoteSubType[2] then -- 'TapNoteSubType_Roll'
				type = '4'
			end
		elseif type:find('Mine') then
			type = 'M'
		else
			type = tostring(TapNoteType:Reverse()[type])
		end
		ndtable[beat] = CharReplace(ndtable[beat], col, type)
		if len > 0 then
			ndtable[beat + len] = CharReplace(ndtable[beat + len], col, '3')
		end
	end
	local ndtablemeasures = {}
	for i, v in ipairs(ndtable) do
		table.insert(ndtablemeasures, v)
		if i % 256 == 0 and i ~= #ndtable then
			table.insert(ndtablemeasures, ',')
		end
	end

	local ndstring = minify(ndtablemeasures)
	--]]
	---[[
	local ndstring
	if filetype == 'ssc' then
		for nd in data:gmatch('#NOTEDATA.-#NOTES2?:[^;]*') do
			nd = nd:gsub('\r\n?', '\n')
			local nd_stepstype, nd_diff, nd_desc = '', '', ''
			for st in nd:gmatch('#STEPSTYPE:(.-);') do
				if nd_stepstype == '' and st ~= '' then
					nd_stepstype = st:gsub('%-', '_'):lower()
					break
				end
			end
			for d in nd:gmatch('#DIFFICULTY:(.-);') do
				if nd_diff == '' and d ~= '' then
					nd_diff = d:lower()
					break
				end
			end
			for d in nd:gmatch('#DESCRIPTION:(.-);') do
				if nd_desc == '' and d ~= '' then
					nd_desc = d
					break
				end
			end
			if (stepstype == nd_stepstype and diff == nd_diff) then
				if (nd_diff ~= 'edit' or nd_desc == desc) then
					ndstring = nd:match("#NOTES2?:[\n]*([^;]*)\n?$"):gsub("//[^\n]*", ""):gsub('[\r\t\f\v ]+', '')
				end
			end
		end
	elseif filetype == 'sm' then
		for nd in data:gmatch('#NOTES2?[^;]*') do
			nd = nd:gsub('\r\n?', '\n')
			local parts = {}
			for part in nd:gmatch('[^:]+') do
				table.insert(parts, part)
			end
			if #parts >= 7 then
				local nd_stepstype = parts[2]:gsub('[^%w-]', ''):gsub('%-', '_'):lower()
				local nd_diff = ToEnumShortString(OldStyleStringToDifficulty(parts[4]:gsub('[^%w]', ''))):lower()
				local nd_desc = parts[3]:gsub('^%s*(.-)', '')
				if (stepstype == nd_stepstype and diff == nd_diff) then
					if (nd_diff ~= 'edit' or nd_desc == desc) then
						ndstring = parts[7]:gsub("//[^\n]*", ""):gsub('[\r\t\f\v ]+', '')
					end
				end
			end
		end
	else
		return
	end
	--]]
	if ndstring and bpms then
		local hash = CRYPTMAN:SHA1String(MinimizeChart(ndstring)..NormalizeFloatDigits(bpms))
		local bytes = {}
		for i = 1, string.len(hash), 1 do
			table.insert(bytes, string.format('%02x', string.byte(hash, i)))
		end
		return table.concat(bytes, ''):sub(1, 16)
	else
		return
	end
end

function gs.request(type, data)
	if not gs.Enabled() then
		return {
			status = 'inactive',
			data = {}
		}
	end
	local id = CRYPTMAN:GenerateRandomUUID()
	local now = GetTimeSinceStart()
	local time = 0
	local timeout = gs.timeout()
	if type == 'ping' then
		id = type
		data = {
			action = 'ping',
			protocol = 1,
		}
	end
	local req = coroutine.create(function(d)
		File.Write(path..'requests/'..id..'.json', JsonEncode(d))
		while true do
			time = time + (GetTimeSinceStart() - now)
			now = GetTimeSinceStart()
			local res = File.Read(path..'responses/'..id..'.json')
			if res then
				coroutine.yield(JsonDecode(res))
				break
			end
			if time > timeout then
				res = {
					status = 'timeout',
					data = {}
				}
				coroutine.yield(res)
				break
			end
		end
	end)
	local s, ret = coroutine.resume(req, data)
	MESSAGEMAN:Broadcast(type, ret)
	return ret
end

function gs.ping()
	return gs.request('Ping', {
		action = 'ping',
		protocol = 1,
	})
end
function gs.session()
	return gs.request('Session', {
		action = 'groovestats/new-session',
		chartHashVersion = 3
	})
end
function gs.scores(data)
	data.action = 'groovestats/player-scores'
	return gs.request('PlayerScores', data)
end
function gs.leaderboards(data)
	data.action = 'groovestats/player-leaderboards'
	return gs.request('PlayerLeaderboards', data)
end
function gs.submit(data)
	data.action = 'groovestats/score-submit'
	return gs.request('ScoreSubmit', data)
end

return gs
