--- This theme will manage high score saving entirely separately from how StepMania insists to do so.
--- Stats.xml will be "ignored" (in reality we're just setting the prefix to something arbitrary so as to create
--- a sort of dummy stats file that won't be used for retrieving what this theme will record as high scores).
--- Instead there will be separate files managed by the theme, stored in the local profile folder.

--- The general overview is that there will be a main stats file which will contain some information similar to how
--- Stats.xml does, but with my own fields and in my own format. There will also be "detailed" high score files
--- which can (later on) be used to recreate all the data associated with that play.

--- Functions dealing with managing the profile data are contained here.

-- these will always be indexed by player number
WF.PlayerProfileStats = {}

-- there will be some checks for the validity of a profile when loaded, and if it's found to be invalid, set a flag to
-- indicate not to try to save anything
WF.ProfileInvalid = {false,false}

-- HERE IT IS BOYS. IT'S THE HASH CACHE #HASHCASH
WF.HashCache = {}

-- this is just my own variable to keep track of how many songs has been played per player
WF.CurrentSessionSongsPlayed = {0,0}

-- subtitle to appear under name on profile card
WF.ProfileCardSubtitle = {"",""}

--[[
PlayerProfileStats table format:
{
    SkillLv = int,
    EP = float,
    SongsPlayed = int,
    TotalTapJudgments = {W1,W2,W3,W4,W5,Miss},
    TotalHolds = {HoldsHeld,HoldsTotal,RollsHeld,RollsTotal},
    TotalMines = {Hit,Total},
    LastSongPlayed = "Group/SongFolder",
    LastPlayedDate = "YYYY-MM-DD",

    -- a couple fields that are not actually stored in the file, but calculated on load
    ClearTypeCounts = {P,EC,GC,FC,HCL,CL,ECL,F},
    GradeCounts = {Ess,AAA,AA,A,B,C,D},
    GradeCounts_ITG = {****,***,**,*,...,F},

    SessionAchievements = {P,EC,GC,FC,HCL,CL,ECL}, -- counts of new clear types this session

    SongStats = {
        -- explanation: this will be a linear array with each item having a ChartID field (the same id i've been using
           for clear types so far) as well as a ChartHash field. There will be a function to search this to find the item
           for a certain chart based on either the ID or the hash. Rate mod will also be an identifier that separates these;
           a chart on a particular rate mod will be treated as a separate chart
        {
            SongFullTitle = "title",
            SongArtist = "artist",
            BPM = "bpm string",
            ChartID = "Style/Group/SongFolder/Difficulty",
            ChartHash = "gs hash",
            RateMod = "1.xx",
            DifficultyRating = int,
            PlayCount = int,
            BestClearType = int,
            BestPercentDP = int (0-10000),
            BestPercentDP_ITG = int,
            Cleared_ITG = "C"/"F", -- nil = unplayed, F = failed, C = cleared
            BestFAPlusCounts = {int,int,int},
            TotalSteps = int,
            TotalHolds = int,
            TotalRolls = int,
            TotalMines = int,
            BestPlay = {
                DateObtained = "YYYY-MM-DD HH:MM:SS",
                Judgments = {W1,W2,W3,W4,W5,Miss,HoldsHeld,RollsHeld,MinesHit},
                FAPlus = {10ms count,12.5ms count},
                LifeBarVals = {Easy,Normal,Hard},
                ScoreAtLifeEmpty = {Easy,Normal,Hard},
                SignificantMods = {"C","Left/Right/Etc","NoMines","NoBoys"}
            }
        },
        ...
        -- Lookup table will allow faster access to any item via SongStats[Lookup[Hash_rate] ] and so forth
        Lookup = {ID_rate=n,Hash_rate=n, ...}
    },
    CourseStats = {
        [TODO] not getting to this yet, lol
    }
}
]]

WF.NewPlayerProfileSongStats = function()
    -- return a blank song stats object for a player profile
    local stats = {
        SongFullTitle = "",
        SongArtist = "",
        BPM = "",
        ChartID = "",
        ChartHash = nil,
        RateMod = "1.0",
        DifficultyRating = 1,
        PlayCount = 0,
        BestClearType = WF.ClearTypes.Fail,
        BestPercentDP = 0,
        BestPercentDP_ITG = 0,
        Cleared_ITG = nil,
        BestFAPlusCounts = {0,0,0},
        TotalSteps = 0,
        TotalHolds = 0,
        TotalRolls = 0,
        TotalMines = 0,
        BestPlay = {
            DateObtained = "0000-00-00 00:00:00",
            Judgments = {0,0,0,0,0,0,0,0,0},
            FAPlus = {0,0},
            LifeBarVals = {1000,1000,1000},
            ScoreAtLifeEmpty = {0,0,0},
            SignificantMods = {}
        }
    }

    return stats
end

WF.AddPlayerProfileSongStatsFromSteps = function(song, steps, rate, hash, pn)
    -- create a new stats object with information from the song and steps passed (and optional rate mod),
    -- and add it to the player profile stats table.
    local player = "PlayerNumber_P"..pn

    if not song then
        song = GAMESTATE:GetCurrentSong()
    end
    if not steps then
        steps = GAMESTATE:GetCurrentSteps(player)
        -- assume if no steps are passed in, everything is based on the current state, so take active rate mod
        rate = RateFromNumber(SL.Global.ActiveModifiers.MusicRate)
    end
    if not rate then rate = "1.0" end
    if (not song) or (not steps) then return end

    local stats = WF.NewPlayerProfileSongStats()
    WF.SongStatsUpdateChartAttributes(stats, song, steps, hash, pn)
    stats.RateMod = rate
    stats.PlayCount = 0
    
    WF.AddSongStatsToProfile(stats, pn)
    return stats
end

WF.SavePlayerProfileStats = function(pn)
    -- Need a player number to get the profile directory (lol)
    if not WF.OKToSaveProfileStats(pn) then return end

    local stats = WF.PlayerProfileStats[pn]
    if (not stats) or stats == {} then return end

    local dir = PROFILEMAN:GetProfileDir("ProfileSlot_Player"..pn)
    if (not dir) or dir == "" then return end

    -- table for all sections of the file
    local ft = {}

    -- first, the general stats
    local statsstr = string.format("#STATS\n%s\n%d\n%f\n%d\n%d,%d,%d,%d,%d,%d\n%d,%d\n%d,%d\n%d,%d\n%s\n%s",
        GetThemeVersion(), stats.SkillLv, stats.EP, stats.SongsPlayed, stats.TotalTapJudgments[1], stats.TotalTapJudgments[2],
        stats.TotalTapJudgments[3], stats.TotalTapJudgments[4], stats.TotalTapJudgments[5], stats.TotalTapJudgments[6],
        stats.TotalHolds[1], stats.TotalHolds[2], stats.TotalHolds[3], stats.TotalHolds[4],
        stats.TotalMines[1], stats.TotalMines[2], stats.LastSongPlayed, stats.LastPlayedDate)
    table.insert(ft, statsstr)

    -- add chart stats for every item in SongStats
    for chart in ivalues(stats.SongStats) do
        -- first, the general stats for the chart
        local hash = chart.ChartHash and chart.ChartHash or "-"
        local itgdp = tostring(chart.BestPercentDP_ITG)..(chart.Cleared_ITG or "")
        table.insert(ft, string.format("#SONG\n%s\n%s\n%s\n%s\n%s\n%s\n%d\n%d\n%d\n%d\n%s\n%d,%d,%d\n%d\n%d,%d,%d",
            chart.SongFullTitle, chart.SongArtist, chart.BPM, hash, chart.ChartID, chart.RateMod,
            chart.DifficultyRating, chart.PlayCount, chart.BestClearType, chart.BestPercentDP, itgdp,
            chart.BestFAPlusCounts[1], chart.BestFAPlusCounts[2], chart.BestFAPlusCounts[3], chart.TotalSteps,
            chart.TotalHolds, chart.TotalRolls, chart.TotalMines))

        -- finally, stats local to the best play on the chart
        local best = chart.BestPlay
        local mods = "-"
        if #best.SignificantMods > 0 then
            mods = table.concat(best.SignificantMods, ",")
        end
        table.insert(ft, string.format("%s\n%d,%d,%d,%d,%d,%d\n%d,%d\n%d\n%d,%d\n%d,%d,%d\n%d,%d,%d\n%s",
            best.DateObtained, best.Judgments[1], best.Judgments[2], best.Judgments[3], best.Judgments[4],
            best.Judgments[5], best.Judgments[6], best.Judgments[7], best.Judgments[8], best.Judgments[9],
            best.FAPlus[1], best.FAPlus[2], best.LifeBarVals[1], best.LifeBarVals[2], best.LifeBarVals[3],
            best.ScoreAtLifeEmpty[1], best.ScoreAtLifeEmpty[2], best.ScoreAtLifeEmpty[3], mods))
    end

    table.insert(ft, "")

    -- write file
    local fstr = table.concat(ft, "\n")
    if File.Write(dir.."/PlayerStats.wfs",fstr) then
        Trace("Player "..pn.." profile stats saved.")
    else
        SM("Player "..pn.." profile stats failed to save!")
    end
end

WF.LoadPlayerProfileStats = function(pn)
    -- Load all the stats from the main stats file
    local dir = PROFILEMAN:GetProfileDir("ProfileSlot_Player"..pn)
    WF.ProfileCardSubtitle[pn] = ""
    if (not dir) or dir == "" then
        -- nil the stats here and exit; this essentially means a guest profile is being used
        WF.PlayerProfileStats[pn] = nil
        return
    end

    local stats = WF.ProfileStatsTemplate()

    -- get subtitle for profile card
    if FILEMAN:DoesFileExist(dir.."/subtitle.txt") then
        WF.ProfileCardSubtitle[pn] = File.Read(dir.."/subtitle.txt")
        -- some light processing, since who knows what people will write in there
        WF.ProfileCardSubtitle[pn] = (WF.ProfileCardSubtitle[pn]:gsub("[\r\f\n]", " ")):sub(1, 20)
    else
        File.Write(dir.."/subtitle.txt", "")
    end

    if not FILEMAN:DoesFileExist(dir.."/PlayerStats.wfs") then
        -- file has not been created; just return
        WF.PlayerProfileStats[pn] = stats
        return
    end

    local sfile = File.Read(dir.."/PlayerStats.wfs")
    if not sfile then
        -- error loading file
        SM("Error loading stats for player "..pn.."!")
        WF.PlayerProfileStats[pn] = nil
        return
    end

    -- line by line logic to read everything from the file into the profile stats
    local lines = split("\n", sfile)
    local token = ""
    local off = 1
    local songind = 0
    local hash = ""
    local id = ""
    for i, line in ipairs(lines) do
        line = line:gsub("[\r\f\n]", "")
        if line == "#STATS" then
            token = "STATS"
            off = i
        elseif line == "#SONG" then
            token = "SONG"
            off = i
            if not stats.SongStats then
                stats.SongStats = { Lookup = {} }
            end
            table.insert(stats.SongStats, {})
            songind = #stats.SongStats
        elseif token == "STATS" then
            local l = i - off
            if l == 1 then
                -- line 1 is wfversion
                if line ~= GetThemeVersion() then
                    -- [TODO] this is where we'd do anything we need to update versions
                end
            elseif l == 2 then
                -- line 2 is skill lv
                stats.SkillLv = tonumber(line)
            elseif l == 3 then
                -- line 3 is EP
                stats.EP = tonumber(line)
            elseif l == 4 then
                -- songs played
                stats.SongsPlayed = tonumber(line)
            elseif l == 5 then
                -- judgment counts
                local judges = line:split_tonumber()
                stats.TotalTapJudgments = {judges[1],judges[2],judges[3],judges[4],judges[5],judges[6]}
            elseif l == 6 then
                -- holds
                local holds = line:split_tonumber()
                stats.TotalHolds = {holds[1],holds[2],0,0}
            elseif l == 7 then
                -- rolls
                local rolls = line:split_tonumber()
                stats.TotalHolds[3] = rolls[1]
                stats.TotalHolds[4] = rolls[2]
            elseif l == 8 then
                -- mines
                local mines = line:split_tonumber()
                stats.TotalMines = {mines[1],mines[2]}
            elseif l == 9 then
                -- last song played
                stats.LastSongPlayed = line
            elseif l == 10 then
                -- last played date
                stats.LastPlayedDate = line
            end
        elseif token == "SONG" then
            local l = i - off
            local song = stats.SongStats[songind]
            if l == 1 then
                -- full title
                song.SongFullTitle = line
            elseif l == 2 then
                -- artist
                song.SongArtist = line
            elseif l == 3 then
                -- bpm string
                song.BPM = line
            elseif l == 4 then
                -- hash
                hash = line
                song.ChartHash = hash ~= "-" and hash or nil
            elseif l == 5 then
                -- id
                id = line
                song.ChartID = id
            elseif l == 6 then
                -- rate mod
                song.RateMod = line

                -- assign lookup index here
                local ext = ""
                if not (song.RateMod == "1.0" or tonumber(song.RateMod) == 1) then
                    ext = "_"..song.RateMod:gsub("%.","_")
                end
                if song.ChartHash then
                    stats.SongStats.Lookup[hash..ext] = songind
                end
                stats.SongStats.Lookup[id..ext] = songind
            elseif l == 7 then
                -- difficulty rating
                song.DifficultyRating = tonumber(line)
            elseif l == 8 then
                -- play count
                song.PlayCount = tonumber(line)
            elseif l == 9 then
                -- clear type
                song.BestClearType = tonumber(line)
            elseif l == 10 then
                -- high score
                song.BestPercentDP = tonumber(line)
            elseif l == 11 then
                -- high itg score
                -- this also contains pass/fail for itg
                local clr = line:match("[FC]") and line:sub(-1)
                local dp = tonumber((line:gsub("[FC]", "")))
                song.BestPercentDP_ITG = dp
                song.Cleared_ITG = clr or (dp ~= 0 and "C")
            elseif l == 12 then
                -- high fa+ counts
                local counts = line:split_tonumber()
                song.BestFAPlusCounts = {counts[1],counts[2],counts[3]}
            elseif l == 13 then
                -- total steps
                song.TotalSteps = tonumber(line)
            elseif l == 14 then
                -- holds, rolls, mines
                local vals = line:split_tonumber()
                song.TotalHolds = vals[1]
                song.TotalRolls = vals[2]
                song.TotalMines = vals[3]
            elseif l == 15 then
                -- best play section; date obtained
                song.BestPlay = {}
                song.BestPlay.DateObtained = line
            elseif l == 16 then
                -- tap judgments
                local counts = line:split_tonumber()
                song.BestPlay.Judgments = {counts[1],counts[2],counts[3],counts[4],counts[5],counts[6],0,0,0}
            elseif l == 17 then
                -- hold judgments
                local held = line:split_tonumber()
                song.BestPlay.Judgments[7] = held[1]
                song.BestPlay.Judgments[8] = held[2]
            elseif l == 18 then
                -- mine hits
                song.BestPlay.Judgments[9] = tonumber(line)
            elseif l == 19 then
                -- fa+
                local counts = line:split_tonumber()
                song.BestPlay.FAPlus = {counts[1],counts[2]}
            elseif l == 20 then
                -- lifebar vals
                local vals = line:split_tonumber()
                song.BestPlay.LifeBarVals = {vals[1],vals[2],vals[3]}
            elseif l == 21 then
                -- score at life empty
                local vals = line:split_tonumber()
                song.BestPlay.ScoreAtLifeEmpty = {vals[1],vals[2],vals[3]}
            elseif l == 22 then
                -- significant mods
                local mods = split(",", line)
                song.BestPlay.SignificantMods = {}
                if line ~= "-" then
                    for mod in ivalues(mods) do
                        table.insert(song.BestPlay.SignificantMods, mod)
                    end
                end
            end
        end
    end

    WF.PlayerProfileStats[pn] = stats

    -- do aggregate math stuff
    WF.CalculateClearTypeAndGradeCounts(pn)

    Trace("Player "..pn.." profile stats loaded.")
end

WF.CalculateClearTypeAndGradeCounts = function(pn)
    -- This calculation will be done on loading profile stats. Call it after everything has loaded from the file.
    local stats = WF.PlayerProfileStats[pn]
    if not stats then return end

    stats.ClearTypeCounts = {0,0,0,0,0,0,0,0}
    stats.GradeCounts = {0,0,0,0,0,0,0}
    stats.GradeCounts_ITG = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}

    for song in ivalues(stats.SongStats) do
        stats.ClearTypeCounts[song.BestClearType] = stats.ClearTypeCounts[song.BestClearType] + 1
        local grade = CalculateGrade(song.BestPercentDP)
        stats.GradeCounts[grade] = stats.GradeCounts[grade] + 1
        local itggrade = CalculateGradeITG(song)
        if itggrade ~= 99 then stats.GradeCounts_ITG[itggrade] = stats.GradeCounts_ITG[itggrade] + 1 end
    end
end

WF.OKToSaveProfileStats = function(pn)
    -- always check this before saving stuff to the profile stats
    return (not WF.ProfileInvalid[pn]) and (WF.PlayerProfileStats[pn] ~= nil)
end

WF.ProfileStatsTemplate = function()
    local stats = {
        SkillLv = 0,
        EP = 0,
        SongsPlayed = 0,
        TotalTapJudgments = {0,0,0,0,0,0},
        TotalHolds = {0,0,0,0},
        TotalMines = {0,0},
        LastSongPlayed = "3D Movie Maker/Bistro Evil Theme",
        LastPlayedDate = WF.DateString(),

        ClearTypeCounts = {0,0,0,0,0,0,0,0},
        GradeCounts = {0,0,0,0,0,0,0},
        SessionAchievements = {0,0,0,0,0,0,0},

        SongStats = {
            Lookup = {}
        },
        CourseStats = {
            Lookup = {}
        }
    }

    return stats
end

WF.WriteDetailedHighScoreStats = function(pn, hsitem)
    -- This should be called at evaluation. It will directly access the detailed judgments table for the current
    -- song and use it to write to the file.
    -- hsitem should be the newly created high score item for the player profile; it's necessary that one was created
    -- if we're writing a new detailed file, so we can get all the general data from it.
    -- NOTE: that will be a bit different once we think about writing to history, but i'm not doing that yet, hehehe
    if not WF.OKToSaveProfileStats(pn) then return end

    local song = GAMESTATE:GetCurrentSong()
    local steps = GAMESTATE:GetCurrentSteps("PlayerNumber_P"..pn)
    local detailed = SL["P"..pn].Stages.Stats[SL.Global.Stages.PlayedThisGame + 1].detailed_judgments

    local hashorid = hsitem.ChartHash and hsitem.ChartHash or hsitem.ChartID
    local fname = LookupIdentifier(hashorid, hsitem.RateMod):gsub("/", "]]][[[")
    local fullpath = PROFILEMAN:GetProfileDir("ProfileSlot_Player"..pn).."/detailed/"..fname

    local abbrev = {Miss = "M", Held = "H", LetGo = "D", HitMine = "N"}

    -- file string will be built from a table to avoid using .. which is bad on performance
    local ft = {}

    -- general info
    local smods = (#hsitem.BestPlay.SignificantMods > 0) and table.concat(hsitem.BestPlay.SignificantMods, ",") or "-"
    local songstr = string.format("%s\n%s\n%s\n%s\n%s\n%d\n%s\n%s\n%d\n%d\n%d\n%d\n%d\n%d\n%s",
        GetThemeVersion(), hsitem.SongFullTitle, hsitem.SongArtist, hsitem.BPM, hsitem.ChartID, hsitem.DifficultyRating,
        hsitem.BestPlay.DateObtained, hsitem.RateMod, hsitem.TotalSteps, hsitem.TotalHolds, hsitem.BestPlay.Judgments[7],
        hsitem.TotalRolls, hsitem.BestPlay.Judgments[8], hsitem.TotalMines, smods)
    table.insert(ft, songstr)

    -- loop through all judgments and output corresponding data
    for judge in ivalues(detailed) do
        -- {seconds, tns, panels, offset}
        local p = (#judge == 3) and string.format("%d", judge[3]) or table.concat(judge[3], "")
        local off = (#judge == 3 or judge[2] == "Miss") and abbrev[judge[2]] or string.format("%f", judge[4])
        table.insert(ft, string.format("%f;%s;%s", judge[1], off, p))
    end
    
    table.insert(ft, "")

    local fstr = table.concat(ft, "\n")

    -- write file
    if not File.Write(fullpath, fstr) then
        SM("Error writing detailed file!")
    end
end


-- machine stats will be much more limited, basically just lists of high scores
--[[ format
{
    SongStats = {
        -- list of items similar to player profile; each item contains a "machine record list"
        -- consisting of "machine record items"
        {
            SongFullTitle = "title",
            SongArtist = "artist",
            BPM = "bpm string",
            ChartHash = "hash",
            ChartID = "id",
            RateMod = "1.xx",
            DifficultyRating = diff,
            PlayCount = count,
            HighScoreList = {
                {
                    PlayerFullName = "name", PlayerHSName = "REEN", PlayerGuid = "guid",
                    DateObtained = "YYYY-MM-DD HH:MM:SS", PercentDP = (0-10000)
                },
                ...
            },
            HighScoreList_ITG = {
                (same format)
            }
        },
        ...
        Lookup = { (Hash)=ind, (ID)=ind, ... }
    }
}
]]
WF.MaxMachineRecordsPerChart = 10
WF.MachineProfileStatsTemplate = function()
    local stats = {
        SongStats = { Lookup = {} },
        CourseStats = { Lookup = {} }
    }
    return stats
end

WF.NewMachineProfileSongStats = function()
    -- create and return a blank chart stats item, but don't actually add it to the profile table.
    -- we mainly don't want to add it because we need the chart id, hash and rate mod first.
    local stats = {
        SongFullTitle = "",
        SongArtist = "",
        BPM = "",
        ChartHash = nil,
        ChartID = "",
        RateMod = "1.0",
        DifficultyRating = 1,
        PlayCount = 0,
        HighScoreList = {},
        HighScoreList_ITG = {}
    }
    setmetatable(stats.HighScoreList, mt_machinerecordlist)
    setmetatable(stats.HighScoreList_ITG, mt_machinerecordlist)

    return stats
end

WF.AddSongStatsToProfile = function(stats, pn)
    -- this function actually adds it to the profile stats table. assume chart id, hash and rate are finally set.
    -- originally this was just for machine profile, but doing it for player profile is exactly the same just using
    -- a different main stats table, so just pass nil for pn to get machine profile.
    local stable = (pn ~= nil) and WF.PlayerProfileStats[pn] or WF.MachineProfileStats
    if not stable then return end

    table.insert(stable.SongStats, stats)
    local ind = #stable.SongStats
    if stats.ChartHash then
        local hid = LookupIdentifier(stats.ChartHash, stats.RateMod)
        stable.SongStats.Lookup[hid] = ind
    end
    local iid = LookupIdentifier(stats.ChartID, stats.RateMod)
    stable.SongStats.Lookup[iid] = ind
end

WF.SongStatsUpdateChartAttributes = function(stats, song, steps, hash, pn)
    -- reusing this code so putting it here. updates things like title, artist, difficulty, etc based on the steps
    -- passed in. this should actually be called any time a score is achieved because there is some chance the simfile
    -- would have been changed.
    -- this doesn't access the Machine or Player song stats tables, but passing in pn will indicate that it's
    -- a player song stats object, meaning step counts etc will need to be updated too.
    local player
    if pn then player = "PlayerNumber_P"..pn end
    local id = WF.GetStepsID(song, steps)

    -- generate hash if one isn't passed in (better to not parse again if we don't need to)
    if not hash then
        -- #HashCash
        if not WF.HashCache[id] then
            local stype = steps:GetStepsType():gsub("StepsType_",""):lower():gsub("_", "-")
            hash = GenerateHash(steps, stype, ToEnumShortString(steps:GetDifficulty()))
            WF.HashCache[id] = hash
        else
            hash = WF.HashCache[id]
        end
    end

    stats.SongFullTitle = song:GetDisplayFullTitle()
    stats.SongArtist = song:GetDisplayArtist()
    stats.BPM = StringifyDisplayBPMs(player, steps, tonumber(rate)):gsub(" ", "")
    stats.ChartHash = (hash ~= "") and hash or nil
    stats.ChartID = WF.GetStepsID(song, steps)
    stats.DifficultyRating = steps:GetMeter()

    if pn then
        local radar = steps:GetRadarValues(player)
        stats.TotalSteps = radar:GetValue("RadarCategory_TapsAndHolds")
        stats.TotalHolds = radar:GetValue("RadarCategory_Holds")
        stats.TotalRolls = radar:GetValue("RadarCategory_Rolls")
        stats.TotalMines = radar:GetValue("RadarCategory_Mines")
    end
end

WF.AddMachineProfileSongStatsFromSteps = function(song, steps, rate, hash, player)
    -- create a new stats object with information from the song and steps passed (and optional rate mod),
    -- and add it to the machine profile stats table.
    -- player is unused unless steps and/or rate are nil. if either is nil, get them from the player passed.
    if not song then
        song = GAMESTATE:GetCurrentSong()
    end
    if (player) and (not steps) then
        steps = GAMESTATE:GetCurrentSteps(player)
        rate = RateFromNumber(SL.Global.ActiveModifiers.MusicRate)
    end
    if not rate then rate = "1.0" end
    if (not song) or (not steps) then return end

    local stats = WF.NewMachineProfileSongStats()
    WF.SongStatsUpdateChartAttributes(stats, song, steps, hash)
    stats.RateMod = rate
    stats.PlayCount = 0
    
    WF.AddSongStatsToProfile(stats)
    return stats
end

WF.NewMachineRecordItem = function()
    -- return a blank machine record object
    local mr = {
        PlayerFullName = "",
        PlayerHSName = "",
        PlayerGuid = "",
        DateObtained = "0000-00-00 00:00:00",
        PercentDP = 0
    }
    setmetatable(mr, mt_machinerecorditem)

    return mr
end

mt_machinerecorditem = {
    __lt = function(a, b)
        if a.PercentDP ~= b.PercentDP then
            return (a.PercentDP < b.PercentDP)
        else
            return (WF.CompareDateTime(a.DateObtained, b.DateObtained) == -1)
        end
    end,
    __le = function(a, b)
        if a.PercentDP ~= b.PercentDP then
            return (a.PercentDP < b.PercentDP)
        else
            return (WF.CompareDateTime(a.DateObtained, b.DateObtained) ~= 1)
        end
    end,
    __eq = function(a, b)
        return ((a.PercentDP == b.PercentDP) and (WF.CompareDateTime(a.DateObtained, b.DateObtained) == 0))
    end
}

mt_machinerecordlist = {
    __newindex = function(list, ind, item)
        -- basically, any time a new item is added to the high score list, sort the list in descending score order
        -- and remove the lowest from the list if there are more items than the max
        --Trace("INSERTING ITEM "..tostring(item).." INTO "..tostring(list).." AT "..tostring(ind))
        rawset(list, ind, item)
        table.sort(list, function(a, b) return a > b end)
        if #list > WF.MaxMachineRecordsPerChart then
            table.remove(list, WF.MaxMachineRecordsPerChart + 1)
        end
    end
}

WF.SaveMachineProfileStats = function()
    local stats = WF.MachineProfileStats
    if (not stats) or stats == {} then return end

    local dir = PROFILEMAN:GetProfileDir("ProfileSlot_Machine")
    if (not dir) or dir == "" then return end

    local ft = {}

    table.insert(ft, GetThemeVersion())

    -- song stats
    for song in ivalues(stats.SongStats) do
        local hash = song.ChartHash and song.ChartHash or "-"
        table.insert(ft, string.format("#SONG\n%s\n%s\n%s\n%s\n%s\n%s\n%d\n%d", song.SongFullTitle, song.SongArtist,
            song.BPM, hash, song.ChartID, song.RateMod, song.DifficultyRating, song.PlayCount))
        for score in ivalues(song.HighScoreList) do
            table.insert(ft, string.format("$SCORE\n%s\n%s\n%s\n%s\n%d", score.PlayerFullName, score.PlayerHSName,
            score.PlayerGuid, score.DateObtained, score.PercentDP))
        end
        for score in ivalues(song.HighScoreList_ITG) do
            table.insert(ft, string.format("$ITG\n%s\n%s\n%s\n%s\n%d", score.PlayerFullName, score.PlayerHSName,
            score.PlayerGuid, score.DateObtained, score.PercentDP))
        end
    end

    -- [TODO] course stats

    table.insert(ft, "")

    -- write to file
    local fstr = table.concat(ft, "\n")
    if File.Write(dir.."/MachineStats.wfm",fstr) then
        Trace("Machine profile stats saved.")
    else
        SM("Machine profile stats failed to save!")
    end
end

WF.LoadMachineProfileStats = function()
    -- Load all the stats from the main stats file
    local dir = PROFILEMAN:GetProfileDir("ProfileSlot_Machine")
    if (not dir) or dir == "" then
        -- trace error and exit
        SM("No machine profile directory found!")
        return
    end

    WF.MachineProfileStats = WF.MachineProfileStatsTemplate()

    if not FILEMAN:DoesFileExist(dir.."/MachineStats.wfm") then
        -- file has not been created; just return
        return
    end

    local sfile = File.Read(dir.."/MachineStats.wfm")
    if not sfile then
        -- error loading file
        SM("Error loading stats for machine profile!")
        return
    end

    -- line by line logic
    local lines = split("\n", sfile)
    local token = ""
    local off = 1
    local songind = 0
    local hash = ""
    local id = ""
    local curstats
    local score
    for i, line in ipairs(lines) do
        line = line:gsub("[\r\f\n]", "")
        if i == 1 then
            -- this is just the version; nothing needed here yet
        elseif line:match("#.*") or line:match("$.*") then
            token = line:sub(2)
            off = i
        elseif token == "SONG" then
            local l = i - off
            if l == 1 then
                -- new song stats
                curstats = WF.NewMachineProfileSongStats()
                curstats.SongFullTitle = line
            elseif l == 2 then
                curstats.SongArtist = line
            elseif l == 3 then
                curstats.BPM = line
            elseif l == 4 then
                hash = line
                curstats.ChartHash = (hash ~= "-") and hash or nil
            elseif l == 5 then
                id = line
                curstats.ChartID = id
            elseif l == 6 then
                curstats.RateMod = line
            elseif l == 7 then
                curstats.DifficultyRating = tonumber(line)
            elseif l == 8 then
                curstats.PlayCount = tonumber(line)
                WF.AddSongStatsToProfile(curstats)
            end
        elseif token == "SCORE" or token == "ITG" then
            local l = i - off
            if l == 1 then
                -- new record item
                score = WF.NewMachineRecordItem()
                score.PlayerFullName = line
            elseif l == 2 then
                score.PlayerHSName = line
            elseif l == 3 then
                score.PlayerGuid = line
            elseif l == 4 then
                score.DateObtained = line
            elseif l == 5 then
                score.PercentDP = tonumber(line)
                -- insert into corresponding table
                local ext = (token == "ITG") and "_ITG" or ""
                curstats["HighScoreList"..ext][#curstats["HighScoreList"..ext]+1] = score
            end
        end
    end

    Trace("Machine profile stats loaded.")
end


-- functions to easily get the propler song stats item given hash, id or song/steps
WF.FindProfileSongStats = function(hashorid, rate, pn)
    -- pass in either a hash or a chart id. simple logic will determine which to use.
    local identifier = hashorid:match("/") and "ChartID" or "ChartHash"
    
    -- pass no pn for machine profile
    local stats = pn ~= nil and WF.PlayerProfileStats[pn] or WF.MachineProfileStats
    if not stats then return end

    -- default to 1.0
    if not rate then rate = "1.0" end

    local ind = LookupIdentifier(hashorid, rate)

    -- first just check using the lookup table
    if stats.SongStats.Lookup[ind] then
        return stats.SongStats[stats.SongStats.Lookup[ind]]
    end

    -- loop through if for some reason it isn't in the lookup table
    for i, item in ipairs(stats.SongStats) do
        if item[identifier] == hashorid and RatesAreEqual(item.RateMod, rate) then
            -- assign the lookup here to correct (really should also get the other identifier but this is an edge
            -- case already)
            stats.SongStats.Lookup[ind] = i
            return item
        end
    end
    -- if this returns nil, you know to add a new item
end

WF.FindProfileSongStatsFromSteps = function(song, steps, rate, hash, pn)
    local stats
    local id = WF.GetStepsID(song, steps)

    -- check hash first
    -- no hash needs to be passed in, but it's better to pass one in if it already exists somewhere, that way
    -- we aren't parsing the file a bunch of times for no reason
    if not hash then
        -- first check the #HashCash for the hash, genearate it and add it to the #HashCash if it's not there
        if not WF.HashCache[id] then
            local stype = steps:GetStepsType():gsub("StepsType_",""):lower():gsub("_", "-")
            hash = GenerateHash(steps, stype, ToEnumShortString(steps:GetDifficulty()))
            WF.HashCache[id] = hash
        else
            hash = WF.HashCache[id]
        end
    end
    if hash and (hash ~= "") then
        stats = WF.FindProfileSongStats(hash, rate, pn)
    end

    if stats then return stats end

    -- next check by id, and if the hash is valid, update the hash in the item
    if id and (id ~= "") then
        stats = WF.FindProfileSongStats(id, rate, pn)
    end

    if stats and hash and (hash ~= "") then
        stats.ChartHash = hash
        local ind = LookupIdentifier(hash, rate)
        local t = pn ~= nil and WF.PlayerProfileStats[pn] or WF.MachineProfileStats
        t.SongStats.Lookup[ind] = FindInTable(stats, t.SongStats)
    end

    return stats
end


-- The following function will handle all the steps for recording to player and machine profiles at the end of a song
-- Returns HSInfo in the format:
--[[
{
    -- p1
    {
        PlayerSongStats = (song stats for current chart in player profile),
        PlayerSongStats_Old = (copy of song stats before update, for comparisons),
        MachineSongStats = (song stats for current chart in machine profile),
        MachineHSInd = (index of high score obtained after inserting to machine profile),
        MachineHSInd_ITG = (itg high score index)
    },
    -- p2
    ...
}
]]
WF.UpdateProfilesOnEvaluation = function(hashes)
    -- see WF-Scoring for StatsObject (this will definitely be revised and less messy later but it's useful here)
    -- hashes should be a table {p1hash, p2hash} -- indices must be player numbers, so for p2 alone {nil, hash}
    local song = GAMESTATE:GetCurrentSong()
    local players = GAMESTATE:GetHumanPlayers()
    local rate = RateFromNumber(SL.Global.ActiveModifiers.MusicRate)
    local dateobtained = WF.DateTimeString()
    local rt = {}

    -- machine stats arrays
    local mstats = {}
    local mr = {}
    local itgmr = {}

    for player in ivalues(players) do
        local pn = tonumber(player:sub(-1))
        local statsobj = WF.CurrentSongStatsObject[pn]
        local pss = STATSMAN:GetCurStageStats():GetPlayerStageStats(player)
        local steps = GAMESTATE:GetCurrentSteps(player)
        local dp = math.floor(statsobj:GetPercentDP() * 10000)
        local itgdp = tonumber(WF.ITGScore[pn]) * 100
        local itgclr = (not WF.ITGFailed[pn]) and "C" or "F"
        rt[pn] = {}

        -- First update player profile, but only need to if it's not a guest
        if WF.PlayerProfileStats[pn] then
            -- Before we actually get song stats, update general player totals.
            local hash = hashes[pn]
            local stats = WF.PlayerProfileStats[pn]
            stats.SongsPlayed = stats.SongsPlayed + 1
            for i = 1, 5 do
                stats.TotalTapJudgments[i] = stats.TotalTapJudgments[i] + pss:GetTapNoteScores("TapNoteScore_W"..i)
            end
            stats.TotalTapJudgments[6] = stats.TotalTapJudgments[6] + pss:GetTapNoteScores("TapNoteScore_Miss")
            local ra = pss:GetRadarActual()
            local rv = pss:GetRadarPossible()
            stats.TotalHolds[1] = stats.TotalHolds[1] + ra:GetValue("Holds")
            stats.TotalHolds[2] = stats.TotalHolds[2] + rv:GetValue("Holds")
            stats.TotalHolds[3] = stats.TotalHolds[3] + ra:GetValue("Rolls")
            stats.TotalHolds[4] = stats.TotalHolds[4] + rv:GetValue("Rolls")
            stats.TotalMines[1] = stats.TotalMines[1] + pss:GetTapNoteScores("TapNoteScore_HitMine")
            stats.TotalMines[2] = stats.TotalMines[2] + rv:GetValue("Mines")
            stats.LastSongPlayed = song:GetSongDir():gsub("/AdditionalSongs/","",1):gsub("/Songs/","",1):sub(1,-2)

            -- Find stats if exists, otherwise create one.
            local songstats = WF.FindProfileSongStatsFromSteps(song, steps, rate, hash, pn)
            if not songstats then
                songstats = WF.AddPlayerProfileSongStatsFromSteps(song, steps, rate, hash, pn)
                -- tentatively bump up counts for newly created clear type/grade; they get adjusted below
                stats.ClearTypeCounts[WF.ClearTypes.Fail] = stats.ClearTypeCounts[WF.ClearTypes.Fail] + 1
                stats.GradeCounts[7] = stats.GradeCounts[7] + 1
            else
                -- just update attributes
                WF.SongStatsUpdateChartAttributes(songstats, song, steps, hash, pn)
                -- add copy of song stats to return table
                rt[pn].PlayerSongStats_Old = WF.CopySongStats(songstats)
            end

            -- Update bests, etc
            songstats.PlayCount = songstats.PlayCount + 1

            local oldct = songstats.BestClearType
            local newct = statsobj:GetClearType()
            if newct < oldct then
                -- update counts, then reassign
                stats.ClearTypeCounts[oldct] = stats.ClearTypeCounts[oldct] - 1
                stats.ClearTypeCounts[newct] = stats.ClearTypeCounts[newct] + 1
                stats.SessionAchievements[newct] = stats.SessionAchievements[newct] + 1
                songstats.BestClearType = statsobj:GetClearType()
            end

            -- update itg grade count (depends on pass or fail, so can't just use dp increase condition)
            local oldgrade_itg = CalculateGradeITG(songstats)
            if itgdp > songstats.BestPercentDP_ITG then songstats.BestPercentDP_ITG = itgdp end
            if (not songstats.Cleared_ITG) or (songstats.Cleared_ITG == "F") then songstats.Cleared_ITG = itgclr end
            local newgrade_itg = CalculateGradeITG(songstats)
            if newgrade_itg < oldgrade_itg then
                if oldgrade_itg ~= 99 then stats.GradeCounts_ITG[oldgrade_itg] =
                    stats.GradeCounts_ITG[oldgrade_itg] - 1 end
                stats.GradeCounts_ITG[newgrade_itg] = stats.GradeCounts_ITG[newgrade_itg] + 1
            end
            for i = 1, 2 do
                if WF.FAPlusCount[pn][i] > songstats.BestFAPlusCounts[i] then
                    songstats.BestFAPlusCounts[i] = WF.FAPlusCount[pn][i]
                end
            end
            if pss:GetTapNoteScores("TapNoteScore_W1") > songstats.BestFAPlusCounts[3] then
                songstats.BestFAPlusCounts[3] = pss:GetTapNoteScores("TapNoteScore_W1")
            end

            -- Update items in "best play" if percent score is a PB
            if dp >= songstats.BestPercentDP then
                -- update grade counts if needed
                local oldgrade = CalculateGrade(songstats.BestPercentDP)
                local newgrade = CalculateGrade(dp)
                if oldgrade ~= newgrade then
                    stats.GradeCounts[oldgrade] = stats.GradeCounts[oldgrade] - 1
                    stats.GradeCounts[newgrade] = stats.GradeCounts[newgrade] + 1
                end

                songstats.BestPercentDP = dp
                songstats.BestPlay.DateObtained = dateobtained
                for i = 1, 5 do
                    songstats.BestPlay.Judgments[i] = pss:GetTapNoteScores("TapNoteScore_W"..i)
                end
                songstats.BestPlay.Judgments[6] = pss:GetTapNoteScores("TapNoteScore_Miss")
                songstats.BestPlay.Judgments[7] = ra:GetValue("Holds")
                songstats.BestPlay.Judgments[8] = ra:GetValue("Rolls")
                songstats.BestPlay.Judgments[9] = pss:GetTapNoteScores("TapNoteScore_HitMine")
                songstats.BestPlay.FAPlus[1] = WF.FAPlusCount[pn][1]
                songstats.BestPlay.FAPlus[2] = WF.FAPlusCount[pn][2]
                for i = 1, 3 do
                    songstats.BestPlay.LifeBarVals[i] = statsobj.LifeBarVals[i]
                end
                for i = 1, 3 do
                    local saf = (WF.LifeBarValues[pn][i].ScoreAtFail ~= -1) and WF.LifeBarValues[pn][i].ScoreAtFail or dp
                    songstats.BestPlay.ScoreAtLifeEmpty[i] = saf
                end
                songstats.BestPlay.SignificantMods = GetSignificantMods(player)

                -- write detailed stats file
                WF.WriteDetailedHighScoreStats(pn, songstats)
            end

            -- add to return table
            rt[pn].PlayerSongStats = songstats
        end

        -- Next update machine profile stats
        -- find machine stats if available, update attributes, increment play count
        mstats[pn] = WF.FindProfileSongStatsFromSteps(song, steps, rate, hash)
        if not mstats[pn] then
            mstats[pn] = WF.AddMachineProfileSongStatsFromSteps(song, steps, rate, hash, player)
        else
            WF.SongStatsUpdateChartAttributes(mstats[pn], song, steps, hash)
        end
        if not (mstats[1] == mstats[2]) then
            -- mstats 1 and 2 will only ever be the same on the second iteration
            -- so this essentially checks if the same chart was played and play count has already incremented
            mstats[pn].PlayCount = mstats[pn].PlayCount + 1
        end

        -- insert into high score list
        local profile = PROFILEMAN:GetProfile(player)
        local isguest = (not WF.PlayerProfileStats[pn])
        mr[pn] = WF.NewMachineRecordItem()
        mr[pn].PlayerFullName = (not isguest) and profile:GetDisplayName() or "Guest"
        mr[pn].PlayerHSName = (not isguest) and profile:GetLastUsedHighScoreName() or "????"
        mr[pn].PlayerGuid = (not isguest) and profile:GetGUID() or "None"
        mr[pn].DateObtained = dateobtained
        mr[pn].PercentDP = dp
        mstats[pn].HighScoreList[#mstats[pn].HighScoreList+1] = mr[pn]

        -- insert into itg high score list
        itgmr[pn] = WF.NewMachineRecordItem()
        itgmr[pn].PlayerFullName = (not isguest) and profile:GetDisplayName() or "C. Foy"
        itgmr[pn].PlayerHSName = (not isguest) and profile:GetLastUsedHighScoreName() or "CFOY"
        itgmr[pn].PlayerGuid = (not isguest) and profile:GetGUID() or ""
        itgmr[pn].DateObtained = dateobtained
        itgmr[pn].PercentDP = itgdp
        mstats[pn].HighScoreList_ITG[#mstats[pn].HighScoreList_ITG+1] = itgmr[pn]

        rt[pn].MachineSongStats = mstats[pn]
    end

    -- need to get machine record index after both potential scores have been inserted
    for player in ivalues(players) do
        local pn = tonumber(player:sub(-1))
        -- we actually can't just FindInTable here, due to the way the metatable handles equality for
        -- machine records (if both players get the same score at the same time, they'll be considered "equal").
        -- fuckin whoops
        for i, score in ipairs(mstats[pn].HighScoreList) do
            local found = true
            for k, v in pairs(score) do if mr[pn][k] ~= v then found = false end end
            if found then rt[pn].MachineHSInd = i break end
        end
        for i, score in ipairs(mstats[pn].HighScoreList_ITG) do
            local found = true
            for k, v in pairs(score) do if itgmr[pn][k] ~= v then found = false end end
            if found then rt[pn].MachineHSInd_ITG = i break end
        end
    end
    
    return rt
end

WF.CopySongStats = function(hs)
    -- this is used to get a copy of the current ("old") high score item on evaluation so that we can make comparisons
    -- to determine what was improved
    local t = {}
    for k, v in pairs(hs) do
        if k ~= "BestFAPlusCounts" and k ~= "BestPlay" then
            t[k] = v
        elseif k == "BestFAPlusCounts" then
            t[k] = {}
            for i = 1,3 do t[k][i] = v[i] end
        end
    end
    return t
end

WF.ImportCheck = function()
    -- redundant with the below function because i wrote it initially to do it all at once, but now i want
    -- to display a message to let the user know what's going on. so this will just check the condition.
    local mdir = PROFILEMAN:GetProfileDir("ProfileSlot_Machine")
    local rt = {false,false,false} -- player 1, player 2, machine
    rt[3] = (not FILEMAN:DoesFileExist(mdir.."/MachineStats.wfm"))
    local players = GAMESTATE:GetHumanPlayers()
    for player in ivalues(players) do
        local pn = player:sub(-1)
        local dir = PROFILEMAN:GetProfileDir("ProfileSlot_Player"..pn)
        if (dir ~= "") and (not FILEMAN:DoesFileExist(dir.."/PlayerStats.wfs")) then
            rt[tonumber(pn)] = true
        end
    end
    return rt
end

WF.FirstTimeImportStats = function()
    -- This will check, for both player profiles and machine profile, if the WF stats files exist.
    -- If they don't, run through some logic to import stats and scores from Exp-Stats.xml (Standard scores),
    -- Stats.xml (ITG scores), and ECFA-Stats.xml (ITG scores, with tier 2 FA+ counts).
    -- Call this in OffConand on the select play mode screen.
    
    -- check which we should import first, so that we can minimize the amount of SetStatsPrefix jawns
    local mdir = PROFILEMAN:GetProfileDir("ProfileSlot_Machine")
    local importmachine = (not FILEMAN:DoesFileExist(mdir.."/MachineStats.wfm"))
    local players = GAMESTATE:GetHumanPlayers()
    local importplayers = {}
    for player in ivalues(players) do
        local pn = player:sub(-1)
        local dir = PROFILEMAN:GetProfileDir("ProfileSlot_Player"..pn)
        if (dir ~= "") and (not FILEMAN:DoesFileExist(dir.."/PlayerStats.wfs")) then
            table.insert(importplayers, player)
        end
    end
    if (not importmachine) and (#importplayers == 0) then return end

    local function Import(prefix)
        PROFILEMAN:SetStatsPrefix(prefix)
        if importmachine then
            WF.ImportProfileStats()
        end
        for player in ivalues(importplayers) do
            local pn = tonumber(player:sub(-1))
            WF.ImportProfileStats(pn)
        end
    end

    -- import "Exp-" (standard) stats and scores
    Import("Exp-")

    -- import no prefix (ITG, no FA+) stats
    Import("")

    -- import "ECFA-" (ITG, with FA+) stats
    Import("ECFA-")

    -- set stats prefix back to WF, so as not to mess with anything else
    PROFILEMAN:SetStatsPrefix("WF-")
end

WF.ImportProfileStats = function(pn)
    -- actual function that does the importing stuff
    -- pn nil for machine
    local stats = (not pn) and WF.MachineProfileStats or WF.PlayerProfileStats[pn]
    local profile = (not pn) and PROFILEMAN:GetMachineProfile() or PROFILEMAN:GetProfile("PlayerNumber_P"..pn)

    local curprefix = PROFILEMAN:GetStatsPrefix()
    local scoretype = {["Exp-"] = "standard", [""] = "ITG", ["ECFA-"] = "ITG (FA+)"}

    -- status message traced to log
    local name = (not pn) and "Machine" or profile:GetDisplayName()
    local statusbase = "Importing scores for profile: "..name
    local status = statusbase
    Trace(status)

    -- update songs played for player
    -- note for updating any profile fields that aren't scores: we can't trust that these are tracked separately
    -- across different stats.xml files due to the prefix bug, so all we can do is take the value if it's higher
    -- than what's already there, as opposed to trying to total them all together
    if pn then
        stats.SongsPlayed = math.max(stats.SongsPlayed, profile:GetNumTotalSongsPlayed())
    end

    for song in ivalues(SONGMAN:GetAllSongs()) do
        for steps in ivalues(song:GetAllSteps()) do
            -- check for high scores
            local hsl = profile:GetHighScoreListIfExists(song, steps)
            if hsl then
                local scores = hsl:GetHighScores()
                local stepsid = WF.GetStepsID(song, steps)
                status = statusbase.."\nScores found for chart: "..stepsid
                Trace(status)
                local bestplayset = {} -- used below
                
                for scoreind, score in ipairs(scores) do
                    -- first check if song stats already exists, and create it if not.
                    -- need to get the songstats each iteration, because rate might be different (ugh)
                    local rate = GetRateFromModString(score:GetModifiers())
                    local songstats = WF.FindProfileSongStatsFromSteps(song, steps, rate, nil, pn)
                    if not songstats then
                        songstats = (not pn) and WF.AddMachineProfileSongStatsFromSteps(song, steps, rate) or
                            WF.AddPlayerProfileSongStatsFromSteps(song, steps, rate, nil, pn)
                    end

                    if not pn then
                        -- if machine, all we need to do is insert the scores into the list
                        local list = (curprefix == "Exp-") and songstats.HighScoreList or songstats.HighScoreList_ITG
                        local hs = WF.NewMachineRecordItem()
                        hs.PlayerFullName = score:GetName()
                        hs.PlayerHSName = score:GetName()
                        hs.PlayerGuid = "Unknown"
                        hs.DateObtained = tostring(score:GetDate())
                        hs.PercentDP = math.round(score:GetPercentDP() * 10000)
                        list[#list+1] = hs
                    else
                        -- for player, we need to get all the personal best stuff we can detect.
                        if curprefix == "" or curprefix == "ECFA-" then
                            -- if itg, all we care about is the high dp score and pass/fail. for fa+, we can also add W1
                            -- judgments to total W1 judgments, as well as record tier 2 and 3 FA+ counts from those.
                            -- pass/fail
                            if score:GetGrade() ~= "Grade_Failed" then songstats.Cleared_ITG = "C"
                            elseif songstats.Cleared_ITG ~= "C" then songstats.Cleared_ITG = "F" end

                            -- best dp for itg
                            songstats.BestPercentDP_ITG = math.max(songstats.BestPercentDP_ITG,
                                math.round(score:GetPercentDP() * 10000))

                            -- fa+ specific stuff
                            if curprefix == "ECFA-" then
                                local faplus = score:GetTapNoteScore("TapNoteScore_W1")
                                songstats.BestFAPlusCounts[2] = math.max(songstats.BestFAPlusCounts[2], faplus)
                                songstats.BestFAPlusCounts[3] = math.max(songstats.BestFAPlusCounts[3], faplus)
                                stats.TotalTapJudgments[1] = stats.TotalTapJudgments[1] + faplus
                            end
                        elseif curprefix == "Exp-" then
                            -- with exp, we can fill most things out since it mirrors the base metrics here.
                            -- particularly, we can fill out "BestPlay" even though it's possible that the actual
                            -- best play was on itg. but in that case, we'd have no way of knowing what any stats
                            -- are for it, so the most we can do is record it for exp.
                            -- let's make use of this old statsobj thing, wow
                            local statsobj = WF.BuildStatsObj(score)
                            local ct = statsobj:GetClearType()
                            songstats.BestClearType = math.min(songstats.BestClearType, ct)
                            local dp = math.round(statsobj:GetPercentDP() * 10000)
                            songstats.BestPercentDP = math.max(songstats.BestPercentDP, dp)
                            local faplus = statsobj:GetJudgmentCount("W1")
                            songstats.BestFAPlusCounts[3] = math.max(songstats.BestFAPlusCounts[3], faplus)

                            -- tally up total taps
                            for i = 1, 5 do
                                stats.TotalTapJudgments[i] = stats.TotalTapJudgments[i] + statsobj:GetJudgmentCount("W"..i)
                            end
                            stats.TotalTapJudgments[6] = stats.TotalTapJudgments[6] + statsobj:GetJudgmentCount("Miss")

                            -- some kind of convoluted logic for best play with rates
                            if not bestplayset[rate] then
                                local bp = songstats.BestPlay
                                bp.DateObtained = tostring(score:GetDate())
                                for i = 1,5 do bp.Judgments[i] = statsobj:GetJudgmentCount("W"..i) end
                                bp.Judgments[6] = statsobj:GetJudgmentCount("Miss")
                                -- just combine holds held and rolls held into holds held
                                bp.Judgments[7] = statsobj:GetJudgmentCount("Held")
                                bp.Judgments[9] = statsobj:GetJudgmentCount("HitMine")
                                -- lifebar values are impossible to get, but we can set arbitrary ones for
                                -- easy/normal based on pass/fail
                                local failed = statsobj:GetFail()
                                bp.LifeBarVals[3] = 0
                                bp.LifeBarVals[1] = (not failed) and 1000 or 0
                                bp.LifeBarVals[2] = (not failed) and 800 or 0
                                for i = 1,3 do bp.ScoreAtLifeEmpty[i] = dp end
                                -- won't bother with significant mods for this
                                bestplayset[rate] = true
                            end
                        end

                        -- there is no way to reliably differentiate holds from rolls in terms of performance vs
                        -- total, so we'll just ignore those. but we can at least tally mines.
                        local radar = score:GetRadarValues()
                        stats.TotalMines[1] = stats.TotalMines[1] + score:GetTapNoteScore("TapNoteScore_HitMine")
                        stats.TotalMines[2] = stats.TotalMines[2] + radar:GetValue("RadarCategory_Mines")

                        -- update play count
                        songstats.PlayCount = songstats.PlayCount + 1
                    end
                end
            end
        end
    end

    status = statusbase.."\nDone"
    Trace(status)

    -- save stats before next load (upon changing the prefix)
    if not pn then WF.SaveMachineProfileStats()
    else WF.SavePlayerProfileStats(pn) end
end

-- get the actorframe to pass into ProfileCard
WF.ProfileCardLowerAF = function(pn, items)
    -- items can be passed in, otherwise will come from ThemePrefs for the profile.
    -- options are "SongsPlayed", "FCTiers", "FCTiersNoPerfect", "LifeBarClears", "TopGradesITG"
    if not items then items = SL["P"..pn].ActiveModifiers.ProfileCardInfo end
    if not items then items = "SongsPlayed" end

    local af = Def.ActorFrame{}

    if items == "SongsPlayed" then
        af[#af+1] = LoadFont("Common Normal")..{
            Text = "Current Session:",
            InitCommand = function(self) self:y(4):zoom(0.8) end
        }
        af[#af+1] = LoadFont("Common Normal")..{
            Text = string.format("%d song%s", WF.CurrentSessionSongsPlayed[pn],
                (WF.CurrentSessionSongsPlayed[pn] > 1) and "s" or ""),
            InitCommand = function(self) self:zoom(0.8):y(20) end
        }
    elseif items == "FCTiers" or items == "FCTiersNoPerfect" or items == "LifeBarClears" then
        local cto = {FCTiers = 0, FCTiersNoPerfect = 1, LifeBarClears = 4}
        for i = 1, 3 do
            local cti = i + cto[items]
            af[#af+1] = LoadFont("Common Normal")..{
                Text = WF.ClearTypesShort[cti],
                InitCommand = function(self) self:zoom(0.8):diffuse(WF.ClearTypeColor(cti)):xy(-20, 14*(i-1)) end
            }
            af[#af+1] = LoadFont("Common Normal")..{
                Text = tostring(WF.PlayerProfileStats[pn].ClearTypeCounts[cti]),
                InitCommand = function(self) self:zoom(0.8):xy(36, 14*(i-1)):horizalign("right"):maxwidth(36/0.8) end
            }
        end
    elseif items == "TopGradesITG" then
        local positions = {{-34, 2}, {12, 2}, {-34, 24}, {12, 24}}
        for i = 1, 4 do
            af[#af+1] = LoadActor(THEME:GetPathG("", "_GradesSmall/LetterGrade.lua"), {grade = i, itg = true})..{
                OnCommand = function(self) self:zoom(0.2):xy(positions[i][1], positions[i][2]) end
            }
            af[#af+1] = LoadFont("Common Normal")..{
                Text = tostring(WF.PlayerProfileStats[pn].GradeCounts_ITG[i]),
                InitCommand = function(self) self:zoom(0.75):xy(positions[i][1] + 32, positions[i][2]+1)
                    :maxwidth(20/0.75):horizalign("right") end
            }
        end
    end

    return af
end


-- The following is a really dumb hacky idea, entirely to try to maintain "Last song played."
-- Firstly, we'll have a flag that signals to the theme to not do any of the WF file operations for profiles
-- if it's set while the Load/SaveProfileCustom functions run, because there is no need.
-- We'll set this flag on the Game Over screen, and run a function that switches the stats prefix back to blank,
-- then saves the profiles, and then switches it back. This is really dumb I'm sorry.
WF.SwitchPrefixFlag = false
WF.DummySave = function(players)
    if not WF.SwitchPrefixFlag then return end
    PROFILEMAN:SetStatsPrefix("")
    for player in ivalues(players) do
        if WF.PlayerProfileStats[tonumber(player:sub(-1))] then
            PROFILEMAN:SaveProfile(player)
        end
    end
    PROFILEMAN:SetStatsPrefix("WF-")
    WF.SwitchPrefixFlag = false
end


-- some utilities

-- (the following 2 functions were moved from WF-Scoring.lua)
WF.GetStepsID = function(song, steps)
    -- this is the index for any item in the SongStats table, pertaining to any steps
    local stypeid = steps:GetStepsType():gsub("StepsType_","")
    local songid = song:GetSongDir():gsub("/AdditionalSongs/","",1):gsub("/Songs/","",1):sub(1,-2)
    local diffid = steps:GetDifficulty():gsub("Difficulty_","")
    local ext = diffid == "Edit" and WF.GetEditIndex(song, steps) or ""
    diffid = diffid:sub(1,1)
    return stypeid.."/"..songid.."/"..diffid..ext
end

WF.GetEditIndex = function(song, steps)
    -- because of the dumbass way edits work, all edit steps have a difficulty of Difficulty_Edit
    -- since we're indexing using the difficulty name, need to find which edit these steps are, and add a number
    local stype = steps:GetStepsType()
    local i = 1
    for v in ivalues(song:GetStepsByStepsType(stype)) do
        if v:GetDifficulty() == "Difficulty_Edit" then
            if v == steps then
                return i
            end
            i = i + 1
        end
    end
end

function LookupIdentifier(hashorid, rate)
    -- indices for the lookup tables should be just the hash or chart id for 1.0 rate, and [Hash]_n_n or [ID]_n_n
    -- for rate modded charts. a chart on 1.2x rate would have the identifier [Hash]_1_2 or [ID]_1_2
    if IsNormalRate(rate) then return hashorid end

    return hashorid.."_"..(NormalizeRateString(rate):gsub("%.", "_"))
end

function CalculateGrade(score)
    -- score should be int score out of 10000
    for i, v in ipairs(WF.GradePercent) do
        if score >= v then
            return i
        end
    end
end

function CalculateGradeITG(scoreitem)
    -- scoreitem can either be int score /10000 or the whole song stats item.
    -- if a number is passed, assume score is a pass, otherwise check Cleared_ITG field
    -- returns number index of grade; 18 is fail, none is 99
    if not scoreitem then return 99 end

    local pass = true
    local score = scoreitem

    if type(scoreitem) ~= "number" then
        if not scoreitem.Cleared_ITG then return 99 end -- unplayed
        pass = (scoreitem.Cleared_ITG == "C")
        score = scoreitem.BestPercentDP_ITG
    end

    if not pass then return 18 end
    for i, v in ipairs(WF.ITGGradeTiers) do
        if score >= v then
            return i
        end
    end

    return 17 -- #itsa17
end

function GetSignificantMods(player)
    -- "significant mods" are defined as player options that are important to show in some way on the results screen.
    -- currently the types of significant mods are:
    --- "C" - cmod, when a chart has stops or multiple displaybpms
    --- (name) - all turn mods
    --- "Mines" - no mines
    --- "ITG" - simulate itg
    --- "FA[n]" fa+ window (n is either 100, 125 or 150)
    --- "NoBoys" / "BigBoys" - error window
    -- returns a table of strings
    local song = GAMESTATE:GetCurrentSong()
    local steps = GAMESTATE:GetCurrentSteps(player)
    local options = GAMESTATE:GetPlayerState(player):GetPlayerOptions("ModsLevel_Preferred")
    local slmods = SL[ToEnumShortString(player)].ActiveModifiers
    local t = {}

    local stops = steps:GetTimingData():GetStops()
    if (options:CMod()) and ((not steps:IsDisplayBpmConstant()) or (stops and (#stops > 0))) then
        table.insert(t, "C")
    end

    local turns = {"Left","Right","Mirror","Shuffle","SoftShuffle","SuperShuffle","NoMines"}
    for turn in ivalues(turns) do
        if options[turn](options) then
            table.insert(t, turn)
        end
    end

    if slmods.SimulateITGEnv then
        table.insert(t, "ITG")
    end

    if slmods.FAPlus ~= 0 then
        table.insert(t, string.format("FA%d", slmods.FAPlus*10000))
    end

    local what = {{"Disabled","NoBoys"},{"Extended","BigBoys"},{"ECFA","SmallBoys"}}
    for boy in ivalues(what) do
        if math.abs(PREFSMAN:GetPreference("TimingWindowSecondsW5") - WF.GetErrorWindowVal(boy[1])) < 0.00001 then
            table.insert(t, boy[2])
            break
        end
    end

    return t
end

-- these few functions are mostly just me being paranoid about how rates will be written to files
function IsNormalRate(rate)
    return (rate == "1.0" or NormalizeRateString(rate) == "1.0")
end
function RateFromNumber(num)
    if math.abs(1 - num) < 0.0001 then return "1.0" end
    local s = tostring(num)
    if s:len() == 1 then s = s..".0" end
    return s
end
function NormalizeRateString(rate)
    return RateFromNumber(tonumber(rate))
end
function RatesAreEqual(rate1, rate2)
    return ((IsNormalRate(rate1) and IsNormalRate(rate2)) or NormalizeRateString(rate1) == NormalizeRateString(rate2))
end

function GetRateFromModString(str)
    str = str:lower()
    local t = split(",", str)
    for i, v in ipairs(t) do
        if v:find("xmusic") then
            local n = tonumber((v:gsub("xmusic", "")))
            return RateFromNumber(n)
        end
    end
    return "1.0"
end

WF.DateString = function(year, month, day)
    if not year then year = Year() end
    if not month then month = MonthOfYear() + 1 end
    if not day then day = DayOfMonth() end
    return string.format("%d-%02d-%02d", year, month, day)
end

WF.TimeString = function(hour, minute, second)
    if not hour then hour = Hour() end
    if not minute then minute = Minute() end
    if not second then second = Second() end
    return string.format("%02d:%02d:%02d", hour, minute, second)
end

WF.DateTimeString = function(year, month, day, hour, minute, second)
    return WF.DateString(year, month, day).." "..WF.TimeString(hour, minute, second)
end

WF.CompareDateTime = function(s1, s2)
    -- should avoid comparisons between items that aren't the same type -- technically, you could compare
    -- just a date to just a time, and the time would always come out greater unless it was 00:00:00
    -- but logically, we just want to be able to quickly check if some date is before another date,
    -- or some time is before another time.
    -- return -1, 1 or 0 depending on if date 1 is before, after or equivalent to date 2, respectively
    local dt1 = WF.DateTime(s1)
    local dt2 = WF.DateTime(s2)
    if dt1 < dt2 then return -1
    elseif dt1 > dt2 then return 1
    else return 0 end
end

WF.DateTime = function(s)
    -- pass in a string like "2020-05-11" or "06:21:00" or "2028-12-25 17:32:00" and get a datetime table out
    if not s then s = WF.DateTimeString() end

    local dt = { Year = 0, Month = 0, Day = 0, Hour = 0, Minute = 0, Second = 0 }
    local hasdate = s:find("-") and true or false
    local hastime = s:find(":") and true or false
    local datestr, timestr
    if hasdate and hastime then
        local sp = split(" ", s)
        datestr = sp[1]
        timestr = sp[2]
    else
        if not hastime then
            datestr = s
        else
            timestr = s
        end
    end

    if datestr then
        local datevals = datestr:split_tonumber("-")
        dt.Year = datevals[1]
        dt.Month = datevals[2]
        dt.Day = datevals[3]
    end
    if timestr then
        local timevals = timestr:split_tonumber(":")
        dt.Hour = timevals[1]
        dt.Minute = timevals[2]
        dt.Second = timevals[3]
    end

    setmetatable(dt, mt_datetime)
    return dt
end
mt_datetime = {
    __lt = function(a, b)
        if a.Year ~= b.Year then return (a.Year < b.Year)
        elseif a.Month ~= b.Month then return (a.Month < b.Month)
        elseif a.Day ~= b.Day then return (a.Day < b.Day)
        elseif a.Hour ~= b.Hour then return (a.Hour < b.Hour)
        elseif a.Minute ~= b.Minute then return (a.Minute < b.Minute)
        elseif a.Second ~= b.Second then return (a.Second < b.Second) end
        return false
    end,
    __le = function(a, b)
        if a.Year ~= b.Year then return (a.Year < b.Year)
        elseif a.Month ~= b.Month then return (a.Month < b.Month)
        elseif a.Day ~= b.Day then return (a.Day < b.Day)
        elseif a.Hour ~= b.Hour then return (a.Hour < b.Hour)
        elseif a.Minute ~= b.Minute then return (a.Minute < b.Minute)
        elseif a.Second ~= b.Second then return (a.Second < b.Second)
        else return true end
        return false
    end,
    __eq = function(a, b)
        local chk = {"Year","Month","Day","Hour","Minute","Second"}
        for field in ivalues(chk) do
            if a[field] ~= b[field] then return false end
        end
        return true
    end
}

string.split_tonumber = function(str, delim)
    -- assume all items can be converted to numbers
    -- i had to do this enough times in this file to warrant a utility function for it
    if not delim then delim = "," end
    local s = split(delim, str)
    for i = 1, #s do
        s[i] = tonumber(s[i])
    end
    return s
end
