-- Stuff relating to tracking clear types and grades is in here

--- itg judgment/lifebar stuff ---
-- similarly to how SL already tracks offsets, this will store all judgment events that occur and at each time
-- in per player tables. each item should have the format { Judgement, NewLifeValue, MusicTimestamp }
-- note: we actually don't care about note offsets here; i'm not building another scatterplot

-- all itg judgments will be indexed according to this table from 1-9
WF.ITGJudgments = enum_table({ "Fantastic", "Excellent", "Great", "Decent", "Way Off", "Miss", "Held", "Dropped", "Mine" })

-- per player variables
WF.ITGJudgmentData = {{},{}}
WF.ITGJudgmentCounts = {}
WF.ITGLife = {0.5,0.5}
WF.ITGDangerThreshold = 0.25
WF.ITGRegenCombo = {0,0}
WF.ITGFailed = {false,false}
WF.ITGScore = {"0.00","0.00"}
WF.ITGDP = {0,0}
-- MaxDP is the maximum dp possible for the whole chart
WF.ITGMaxDP = {0,0}
-- CurMaxDP is the maximum dp that could be possible to have -right now- in the chart
-- (so increase by 5 every tap and hold). These are needed for subtractive scoring.
WF.ITGCurMaxDP = {0,0}
WF.ITGCombo = {0,0}
WF.ITGFCType = {1,1} -- 1 = FFC, 2 = FEC, 3 = FC, 4 = None

-- call this when screengameplay starts
WF.InitITGTracking = function(pn)
    WF.ITGJudgmentCounts[pn] = { 0, 0, 0, 0, 0, 0, 0, 0, 0 }
    WF.ITGJudgmentData[pn] = {}
    WF.ITGLife[pn] = 0.5
    WF.ITGRegenCombo[pn] = 0
    WF.ITGFailed[pn] = false
    WF.ITGScore[pn] = "0.00"
    WF.ITGDP[pn] = 0
    WF.ITGMaxDP[pn] = WF.GetITGMaxDP("PlayerNumber_P"..pn)
    WF.ITGCurMaxDP[pn] = 0
    WF.ITGCombo[pn] = 0
    WF.ITGFCType[pn] = 1
end

WF.ITGTimingWindows = {
    0.023, -- Fantastic
    0.0445, -- Excellent
    0.1035, -- Great
    0.1365, -- Decent
    0.1815 -- Way Off
}

WF.ITGScoreWeights = {
    5, -- Fantastic
    4, -- Excellent
    2, -- Great
    0, -- Decent
    -6, -- Way Off
    -12, -- Miss
    5, -- Held
    0, -- Dropped
    -6 -- Mine
}
WF.ECFAScoreWeights = {
    3, -- Fantastic
    2, -- Excellent
    1, -- Great
    0, -- Decent
    0, -- Way Off
    0, -- Miss
    0, -- Held
    -1, -- Dropped
    -1 -- Mine
}
WF.ITGLifeChanges = {
    0.008, -- Fantastic
    0.008, -- Excellent
    0.004, -- Great
    0, -- Decent
    -0.05, -- Way Off
    -0.1, -- Miss
    0.008, -- Held
    -0.08, -- Dropped
    -0.05, -- Mine
    MissedHold = 0 -- this should not exist (see below)
}
WF.ITGGradeTiers = {
    10000,
    9900,
    9800,
    9600,
    9400,
    9200,
    8900,
    8600,
    8300,
    8000,
    7600,
    7200,
    6800,
    6400,
    6000,
    5500
}
WF.GetITGGrade = function(score)
    -- pass in the formatted nn.nn string as the score. this assumes score is a pass.
    local s = tonumber(score) * 100
    for i, v in ipairs(WF.ITGGradeTiers) do
        if s >= v then
            return string.format("%02d", i)
        end
    end

    return "17" -- #itsa17
end

WF.GetITGJudgment = function(offset)
    for i, v in ipairs(WF.ITGTimingWindows) do
        if math.abs(offset) <= v then
            return i
        end
    end

    return WF.ITGJudgments.Miss -- this should never happen but who knows
end

WF.TrackITGJudgment = function(pn, judgedata)
    -- judgedata should be the params table from the JudgmentMessageCommand

    -- exit if autoplay
    if GAMESTATE:GetPlayerState("PlayerNumber_P"..pn):GetPlayerController() == "PlayerController_Autoplay" then
        return
    end

    -- use ecfa weights if in ecfa mode
    local weights = (WF.ECFAMatch == "None") and WF.ITGScoreWeights or WF.ECFAScoreWeights

    local songtime = GAMESTATE:GetCurMusicSeconds()
    local j = -1
    if judgedata.TapNoteScore and not judgedata.HoldNoteScore then
        if judgedata.TapNoteScore == "TapNoteScore_AvoidMine" then return end -- we don't care about dodging mines
        if judgedata.TapNoteScore == "TapNoteScore_HitMine" then
            j = WF.ITGJudgments.Mine
        else
            j = judgedata.TapNoteScore ~= "TapNoteScore_Miss" and WF.GetITGJudgment(judgedata.TapNoteOffset) or WF.ITGJudgments.Miss
        end
    elseif judgedata.HoldNoteScore then
        if judgedata.HoldNoteScore == "HoldNoteScore_Held" then
            j = WF.ITGJudgments.Held
        elseif judgedata.HoldNoteScore == "HoldNoteScore_LetGo" then
            j = WF.ITGJudgments.Dropped
        elseif judgedata.HoldNoteScore == "HoldNoteScore_MissedHold" then
           -- there is actually another HoldNoteScore (MissedHold) that happens at the end of a hold when the note was missed
           -- incidentally, we actually have to track this judgment because it decrements combotoregainlife
           -- this seems... unintended, but it happens
           -- update: this is indeed a bug and is not how "real" itg behaves. so i'm ignoring it here.
           -- update 2: as it turns out, we need this for subtractive scoring. so just take care of that here.
           WF.ITGCurMaxDP[pn] = WF.ITGCurMaxDP[pn] + weights[WF.ITGJudgments.Held]
           return --j = "MissedHold" 
        end
    end

    if j ~= -1 then
        -- most things we don't need to do if you've already failed
        if not WF.ITGFailed[pn] then
            -- update dp first
            WF.ITGDP[pn] = WF.ITGDP[pn] + weights[j]
            -- update max dp -- conveniently, both held and fantastic are +5, so we can increase it by 5 for any
            -- judgment except HitMine :)
            -- need to actually handle this separately for ecfa lol
            if j < 9 then
                WF.ITGCurMaxDP[pn] = WF.ITGCurMaxDP[pn] + (j <= 6 and weights[WF.ITGJudgments.Fantastic] or weights[WF.ITGJudgments.Held])
            end

            -- track judgments in table
            local newlife = WF.UpdateLifeValue(pn, j)
            table.insert(WF.ITGJudgmentData[pn], { j, newlife, songtime })
        end

        -- combo stuff
        if j <= 3 then
            WF.ITGCombo[pn] = WF.ITGCombo[pn] + 1
            if WF.ITGFCType[pn] < j then
                WF.ITGFCType[pn] = j
            end
        else
            if (j >= WF.ITGJudgments.Decent and j <= WF.ITGJudgments.Miss) or j == WF.ITGJudgments.Dropped then
                WF.ITGFCType[pn] = 4
                if j ~= WF.ITGJudgments.Dropped then
                    WF.ITGCombo[pn] = 0 -- this condition was always strange to me haha
                end
            end
        end
    end
end

WF.UpdateLifeValue = function(pn, judgment)
    -- judgment should be the actual judgment index based on the table defined in this script
    --- ugh, you don't really realize how annoying this lifebar is until you're remaking it, do you

    -- if failed already, just return a 0
    if WF.ITGFailed[pn] then return 0 end

    local oldlife = WF.ITGLife[pn]
    
    -- update regencombo first, then change life accordingly
    if WF.ITGLifeChanges[judgment] < 0 then
        WF.ITGRegenCombo[pn] = math.min(WF.ITGRegenCombo[pn] + 5, 10)
    else
        WF.ITGRegenCombo[pn] = math.max(WF.ITGRegenCombo[pn] - 1, 0)
    end

    if not (WF.ITGRegenCombo[pn] > 0 and WF.ITGLifeChanges[judgment] > 0) then
        -- harsh hot life penalty, god
        local lifechange = (WF.ITGLife[pn] == 1 and WF.ITGLifeChanges[judgment] < 0) and -0.1 or WF.ITGLifeChanges[judgment]
        WF.ITGLife[pn] = math.min(WF.ITGLife[pn] + lifechange, 1)
        if WF.ITGLife[pn] <= 0.00001 then
            -- set fail state here
            WF.ITGLife[pn] = 0
            WF.ITGFailed[pn] = true
            MESSAGEMAN:Broadcast("ITGFailed", { pn = pn })
            --SCREENMAN:SystemMessage("failed")
        end
        --local h = SCREENMAN:GetTopScreen():GetChild("LifeP"..pn):GetLife()
        --if not WF.ITGFailed[pn] then SCREENMAN:SystemMessage(tostring(WF.ITGLife[pn]).."  "..h.."  "..judgment) end
    else
        --SCREENMAN:SystemMessage("ctrl "..WF.ITGRegenCombo[pn].."  "..judgment)
    end

    -- broadcast a message if life changed
    if WF.ITGLife[pn] ~= oldlife then
        MESSAGEMAN:Broadcast("ITGLifeChanged", {pn = pn, oldlife = oldlife, newlife = WF.ITGLife[pn]})
        if (WF.ITGLife[pn] <= WF.ITGDangerThreshold and oldlife > WF.ITGDangerThreshold) then
            MESSAGEMAN:Broadcast("ITGDanger", {pn = pn, event = "In"})
        elseif (WF.ITGLife[pn] > WF.ITGDangerThreshold and oldlife <= WF.ITGDangerThreshold) then
            MESSAGEMAN:Broadcast("ITGDanger", {pn = pn, event = "Out"})
        elseif (WF.ITGLife[pn] <= 0) then
            MESSAGEMAN:Broadcast("ITGDanger", {pn = pn, event = "Dead"})
        end
    end

    return WF.ITGLife[pn] -- TrackITGJudgment will use the new life value
end

WF.ConsolidateJudgments = function(pn)
    -- build a table of judgment counts, because that's easier to work with (:
    -- call this first on the evaluation screen
    WF.ITGJudgmentCounts[pn] = { 0, 0, 0, 0, 0, 0, 0, 0, 0 }

    for i, v in ipairs(WF.ITGJudgmentData[pn]) do
        WF.ITGJudgmentCounts[pn][v[1]] = WF.ITGJudgmentCounts[pn][v[1]] + 1
    end
end

WF.GetITGMaxDP = function(player)
    local weights = (WF.ECFAMatch == "None") and WF.ITGScoreWeights or WF.ECFAScoreWeights
    local radar = GAMESTATE:GetCurrentSteps(player):GetRadarValues(player)
    local totalholdjudgments = radar:GetValue("RadarCategory_Holds") + radar:GetValue("RadarCategory_Rolls")
    local totaltapjudgments = radar:GetValue("RadarCategory_TapsAndHolds")
    return totalholdjudgments * weights[WF.ITGJudgments.Held]
        + totaltapjudgments * weights[WF.ITGJudgments.Fantastic]
end

WF.GetITGPercentDP = function(player, maxdp)
    -- if maxdp is passed in, just use that so we don't have to call current steps every time
    if not maxdp then maxdp = WF.GetITGMaxDP(player) end

    if maxdp == 0 then return 0 end

    local pn = tonumber(player:sub(-1))
    local raw = WF.ITGDP[pn] / maxdp
    return math.max(0, math.floor(raw * 10000) / 10000)
end

WF.CalculateITGScore = function(player)
    -- call this on ScreenEvaluation
    -- this will return the score value as well as set the global WF.ITGScore[pn]
    local pn = tonumber(player:sub(-1))
    local pss = STATSMAN:GetCurStageStats():GetPlayerStageStats(player)
    local weights = (WF.ECFAMatch == "None") and WF.ITGScoreWeights or WF.ECFAScoreWeights

    -- get possible/actual dp
    local totalholdjudgments = pss:GetRadarPossible():GetValue("RadarCategory_Holds") + pss:GetRadarPossible():GetValue("RadarCategory_Rolls")
    local totaltapjudgments = pss:GetRadarPossible():GetValue("RadarCategory_TapsAndHolds")
    local possibledp = totalholdjudgments * weights[WF.ITGJudgments.Held]
        + totaltapjudgments * weights[WF.ITGJudgments.Fantastic]
    local dp = 0
    for i, v in ipairs(WF.ITGJudgmentCounts[pn]) do
        dp = dp + v * weights[i]
    end

    if possibledp == 0 then return "0.00" end

    local rawscore = math.max(dp / possibledp, 0)
    -- return formatted string
    WF.ITGScore[pn] = string.format("%.2f", math.floor(rawscore * 10000) / 100)
    return WF.ITGScore[pn]
end

-- calculate vertices for the itg lifebar graph (wow!)
--- the logic here is, there doesn't need to be more than [graph width] vertices. calculate the amount of time in the song
--- would take up 1 pixel of graph, and then for every interval between judgments, if the gap is larger than 1 pixel,
--- use the raw judgment for the next vertex.
--- originally i was going to average values between pixels but in practice that doesn't look necessary at all -- in fact,
--- my lifebar graph as it is is more precise/granular than the built in graph in game so checkmate atheists
WF.GetITGLifeVertices = function(pn, graphwidth, graphheight, songstart, songend)
    local verts = {}
    local songlength = songend - songstart
    local timescale = songlength / graphwidth
    local t = songstart
    local l = 0.5

    table.insert(verts, {{0,graphheight/2,0},{1,1,1,1}})
    if (#WF.ITGJudgmentData[pn] > 0) then
        table.insert(verts, {{(WF.ITGJudgmentData[pn][1][3] - songstart) / timescale, graphheight/2, 0},{1,1,1,1}})
    end

    for i, v in ipairs(WF.ITGJudgmentData[pn]) do
        if v[3] - t >= timescale or v[2] < 0.0001 then
            table.insert(verts, {{(v[3] - songstart) / timescale, (1 - v[2]) * graphheight, 0},{1,1,1,1}})
            t = v[3]
        end

        l = v[2]
    end

    table.insert(verts, {{graphwidth, (1 - l) * graphheight, 0},{1,1,1,1}})

    return verts
end


--- clear type/award stuff ---

-- grade tiers
-- "grades" will actually be overridden with my own grades, since SM doesn't allow a failing score to not
-- have a grade of grade_failed (why)
WF.Grades = enum_table({"S","AAA","AA","A","B","C","D"})
WF.GradePercent = {
    9900, -- S
    9700, -- AAA
    9500, -- AA
    9000, -- A
    8000, -- B
    7000, -- C
    -999  -- anything under 70 would give a grade of D
}

-- clear types
WF.ClearTypes = enum_table({
    "Perfection",
    "Excellent Combo",
    "Great Combo",
    "Full Combo",
    WF.LifeBarNames[3].." Clear",
    "Clear",
    WF.LifeBarNames[1].." Clear",
    "Fail"
})

WF.ClearTypesShort = enum_table({
    "Perfect",
    "EC",
    "GC",
    "FC",
    "HCL",
    "CL",
    "ECL",
    "Fail"
})

WF.ClearTypeColor = function(ct)
    if (not ct) or ct == 0 then return Color.White end
    if type(ct) == "string" then ct = WF.ClearTypes[ct] end
    if ct <= 4 then
        return SL.JudgmentColors.Waterfall[ct]
    elseif ct < 8 then
        return WF.LifeBarColors[8 - ct]
    else
        return color("#B00000")
    end
end

-- for the purpose of keeping track of play history etc, we want to be able to easily build a stats object
-- that has universal functions for GetTapNotes etc
-- the idea is that you can easily get all this information whether you're passing in a PlayerStageStats object
-- on ScreenEvaluation, or a HighScore object when loading the profile, or anywhere else

WF.StatsObj = {
    __index = function(self, arg)
        return WF.StatsObj[arg]
    end,

    GetType = function(self)
        -- determine the type of the object by checking functions that only exist in any particular one
        if self.Stats.FailPlayer then
            return "PlayerStageStats"
        elseif self.Stats.GetDate then
            return "HighScore"
        else
            return "Custom"
        end
    end,

    GetTapNotes = function(self, arg)
        if not arg:find("TapNoteScore") then
            arg = "TapNoteScore_"..arg
        end
        if self:GetType() == "PlayerStageStats" then
            return self.Stats:GetTapNoteScores(arg)
        elseif self:GetType() == "HighScore" then
            return self.Stats:GetTapNoteScore(arg)
        else
            return self.Stats[arg]
        end
    end,

    GetHoldNotes = function(self, arg)
        if not arg:find("HoldNoteScore") then
            arg = "HoldNoteScore_"..arg
        end
        if self:GetType() == "PlayerStageStats" then
            return self.Stats:GetHoldNoteScores(arg)
        elseif self:GetType() == "HighScore" then
            return self.Stats:GetHoldNoteScore(arg)
        else
            return self.Stats[arg]
        end
    end,

    GetJudgmentCount = function(self, arg)
        local tapnames = {"W1","W2","W3","W4","W5","Miss","HitMine","AvoidMine"}
        local holdnames = {"Held","LetGo","MissedHold"}
        if FindInTable(arg, tapnames) or arg:find("TapNoteScore") then
            return self:GetTapNotes(arg)
        end
        if FindInTable(arg, holdnames) or arg:find("HoldNoteScore") then
            return self:GetHoldNotes(arg)
        end
    end,

    GetSkipped = function(self)
        -- skipping to results is not tracked in ITG/SL but we don't want to give a clear if we can detect
        -- that the song was skipped. if a PSS is passed in this is simple, but with a HighScore we need
        -- to validate the DP percent, buh. we shouldn't need to care about this for custom table because
        -- we can set the lifebar values to 0 in real time when a song is skipped
        if self.LifeBarVals then
            -- before doing anything, we don't care if the song was skipped if we know all lifebars were 0
            -- (ie the song was failed and then skipped to the results screen)
            local f = true
            for v in ivalues(self.LifeBarVals) do
                if v > 0 then
                    f = false
                    break
                end
            end
            if f then
                return false
            end
        end
        local totaltaps = self:GetJudgmentCount("W1") + self:GetJudgmentCount("W2") + self:GetJudgmentCount("W3")
            + self:GetJudgmentCount("W4") + self:GetJudgmentCount("W5") + self:GetJudgmentCount("Miss")
        local totalholds = self:GetJudgmentCount("Held") + self:GetJudgmentCount("LetGo") + self:GetJudgmentCount("MissedHold")
        if self:GetType() == "PlayerStageStats" then
            local rv = self.Stats:GetRadarPossible()
            return not (totaltaps == rv:GetValue("RadarCategory_TapsAndHolds")
                        and totalholds == rv:GetValue("RadarCategory_Holds") + rv:GetValue("RadarCategory_Rolls"))
        elseif self:GetType() == "HighScore" then
            return (not self:ValidateDP())
        end
    end,

    GetFail = function(self)
        -- since we're using custom lifebars, life bar values can be passed in separately. if they arent, use
        -- the PSS or HS object to get a normal clear or fail
        if self.LifeBarVals ~= nil and not self:GetSkipped() then
            for i, v in ipairs(self.LifeBarVals) do
                if v > 0 then return false end
            end
            return true
        elseif self:GetType() == "PlayerStageStats" then
            return (self.Stats:GetFailed() or self:GetSkipped())
        elseif self:GetType() == "HighScore" then
            return (self.Stats:GetGrade() == "Grade_Failed" or self:GetSkipped())
        end
    end,

    GetPercentDP = function(self)
        -- used in ValidateDP; for a HighScore/PSS, just return the % dp contained in the object, otherwise calculate
        if self:GetType() == "HighScore" then
            return self.Stats:GetPercentDP()
        elseif self:GetType() == "PlayerStageStats" then
            return self.Stats:GetPercentDancePoints()
        else
            -- calculate for custom table
            return self:CalculateDP()
        end
    end,

    CalculateDP = function(self)
        -- different from the above in that it will always do the calculation. ValidateDP will compare these two.
        ---- NOTE: using "SL" table here to get score weights, remember to change that later ----
        local m = SL.Metrics.Waterfall
        local totaltaps = self:GetJudgmentCount("W1") + self:GetJudgmentCount("W2") + self:GetJudgmentCount("W3")
            + self:GetJudgmentCount("W4") + self:GetJudgmentCount("W5") + self:GetJudgmentCount("Miss")
        local totalholds = self:GetJudgmentCount("Held") + self:GetJudgmentCount("LetGo") + self:GetJudgmentCount("MissedHold")

        if totaltaps + totalholds == 0 then
            return 0 --don't divide by 0
        end

        local dp = 0
        local j = {"W1","W2","W3","W4","W5","Miss","Held","LetGo","HitMine"}
        for v in ivalues(j) do
            dp = dp + m["PercentScoreWeight"..v] * self:GetJudgmentCount(v)
        end

        return math.max(0, dp / (totaltaps * m.PercentScoreWeightW1 + totalholds * m.PercentScoreWeightHeld))
    end,

    ValidateDP = function(self)
        --Trace("Validating dp... actual "..self:GetPercentDP().." calculated "..self:CalculateDP())
        return math.abs(self:GetPercentDP() - self:CalculateDP()) <= 0.0001
    end,

    GetScoreString = function(self)
        return FormatPercentScore(self:GetPercentDP()):gsub("%%","")
        --return string.format("%.2f", math.floor(self:GetPercentDP() * 10000) / 100)
    end,

    GetClearType = function(self)
        -- we can use the judgment breakdowns to derive all "full combo" types, but for lifebar clears, we can only
        -- derive approximated "normal clear" without passing in lifebar values (ie in a HighScore on loading profile)

        -- return fail if skipped
        if self:GetSkipped() then return WF.ClearTypes.Fail end

        if self:GetJudgmentCount("Miss") > 0 or self:GetJudgmentCount("LetGo") > 0 or self:GetJudgmentCount("HitMine") > 0
        or self:GetJudgmentCount("W5") > 0 then
            -- not a full combo; just get pass/fail
            if not self.LifeBarVals then
                return self:GetFail() and WF.ClearTypes.Fail or WF.ClearTypes.Clear
            else
                for i = #self.LifeBarVals, 1, -1 do
                    if self.LifeBarVals[i] > 0 then
                        return i == WF.LifeBarNames.Normal and WF.ClearTypes.Clear
                            or WF.ClearTypes[WF.LifeBarNames[i].." Clear"] -- i guess it works
                    end
                end
                return WF.ClearTypes.Fail
            end
        else
            -- full combo tiers
            for i = 4, 1, -1 do
                if self:GetJudgmentCount("W"..i) > 0 then
                    return i -- maybe a little weird but we know that the first 4 clear types correspond to the windows
                end
            end
        end

        -- it's possible to get here if for some reason no judgments happened at all. give em a hard clear i guess
        return WF.ClearTypes[WF.LifeBarNames[3].." Clear"]
    end,

    CalculateGrade = function(self)
        -- just the calculation for a grade
        local score = tonumber((self:GetScoreString())) * 100
        for i, v in ipairs(WF.GradePercent) do
            if score >= v then
                return i
            end
        end
    end,

    GetGrade = function(self)
        -- we actually don't want to use the "F" grade at all, because "fail" is categorized under clear types
        -- so for a PSS, we need to override a failing grade by calculating from the score
        if self:GetType() == "PlayerStageStats" and not self:GetFail() then
            return tonumber(self.Stats:GetGrade():sub(-1))
        else
            return self:CalculateGrade()
        end
    end
}

-- we will use this player indexed array on evaluation to pass into profile stuff
WF.CurrentSongStatsObject = {}

WF.BuildStatsObj = function(stats, lifebarvals)
    -- stats can either be a PlayerStageStats, a HighScore or a list of judgments in the format
    -- {TapNoteScore_W1 = n, ... HoldNoteScore_Held = n, ... }
    -- lifebarvals is an optional list of ending lifebar values from easy to hard {n,n,n}
    local t = { Stats = stats, LifeBarVals = lifebarvals }
    setmetatable(t, WF.StatsObj)
    return t
end


-- Function to consolidate per panel judgments and apply the counts to the SL data table
WF.ConsolidatePerPanelJudgments = function(pn, judgments)
    -- judgments should be the detailed_judgments table recorded from ScreenGameplay
    local coljudgments = SL["P"..pn].Stages.Stats[SL.Global.Stages.PlayedThisGame + 1].column_judgments

    if not coljudgments then 
        coljudgments = {}
        for i = 1,8 do table.insert(coljudgments, {W1=0,W2=0,W3=0,W4=0,W5=0,Miss=0,MissBecauseHeld=0}) end
    end

    for i, jdata in ipairs(judgments) do
        if not (jdata[2] == "Held" or jdata[2] == "LetGo" or jdata[2] == "HitMine") then
            for c in ivalues(jdata[3]) do
                coljudgments[c][jdata[2]] = coljudgments[c][jdata[2]] + 1
            end
        end
    end
end


--- EARTH POWER STUFF ---
-- this stuff is just kinda sitting here; i was messing with it before but what i am going to do for EP is going
-- to be much different
WF.EPBase20Val = 5000
WF.EPStepCountScale = function(stepcount)
    --return -1.69810951 + (0.335333363 * math.log(stepcount))
    return 0.5706947392 * (1.000041868^stepcount)
end

WF.GetSustainedDensity = function(song, steps, measures)
    -- gets max NPS for intervals of the given amount of measures long
    -- returns 0 if the chart is fewer measures than the number given
    local nps, dt = GetNPSperMeasure(song, steps)
    if #dt < measures then
        return 0
    end

    local density = 0
    local sum = 0
    for i, v in ipairs(dt) do
        if not dt[i + measures - 1] then
            break
        end
        -- sum up the elements if this is the first iteration, but otherwise just subtract the last one and add the next one
        -- this way we only have to loop through i-i+measures-1 once
        if i == 1 then
            for j = 1, measures do
                sum = sum + dt[j]
            end
        else
            sum = sum - dt[i-1] + dt[i + measures - 1]
        end
        local avgd = sum / measures
        if avgd > density then
            density = avgd
        end
    end

    return density
end

WF.EPDensityScale = function(density)
    return 0.0251723461 * (1.307487649^density)
    --return 0.03569800644 * (1.292664602^density)
end

WF.EPSustainedDensityScale = function(density)
    --return 0.03569800644 * (1.292664602^density)
    return 0.0251723461 * (1.307487649^density)
end

WF.CalculateBaseEP = function(song, steps, player)
    -- why you need to pass in a player to get chart stepcount is beyond me?? fuck you
    local stepcount = steps:GetRadarValues(player):GetValue("RadarCategory_TapsAndHolds")
    local length = song:GetLastSecond() - song:GetFirstSecond()
    local density = stepcount / length
    local sdensity = WF.GetSustainedDensity(song, steps, 12)
    local sd_f = math.max(0, 1 - density/sdensity)
    local d_factor = WF.EPDensityScale(density) + sd_f * WF.EPSustainedDensityScale(sdensity)
    return WF.EPStepCountScale(stepcount) * d_factor * WF.EPBase20Val
end


--- Function for setting the timing window based on the ExpErrorWindow setting
WF.SetErrorWindow = function(setting)
    -- pass the string value of the option here
    -- special condition "ECFA" will set the error window to the size of ITG great
    local val = WF.GetErrorWindowVal(setting)
    if val ~= nil and PREFSMAN:GetPreference("TimingWindowSecondsW5") ~= val then
        PREFSMAN:SetPreference("TimingWindowSecondsW5", val)
        -- hijack "TimingWindow" modifier to indicate to SL that W5 is disabled, if that option is selected
        SL.Global.ActiveModifiers.TimingWindows[5] = (setting ~= "Disabled")
    end
    --SCREENMAN:SystemMessage(tostring(PREFSMAN:GetPreference("TimingWindowSecondsW5")))
end

WF.GetErrorWindowVal = function(setting)
    if setting == "Enabled" then
        return SL.Preferences.Waterfall.TimingWindowSecondsW5
    elseif setting == "Disabled" then
        return 0
    elseif setting == "Extended" then
        return SL.Preferences.ITG.TimingWindowSecondsW5 + SL.Preferences.ITG.TimingWindowAdd
    elseif setting == "ECFA" then
        return SL.Preferences.ITG.TimingWindowSecondsW3 + SL.Preferences.ITG.TimingWindowAdd
    end
end


-- FA+ tracking
-- This will be per player, and indexed by [1] = 10ms count, [2] = 12.5ms count
-- We actually don't need to track 15ms because the Perfect window is always 15ms
WF.FAPlusCount = {}
WF.InitFAPlus = function(pn)
    -- Call this at the start of ScreenGameplay (probably in offset tracking)
    -- [TODO] need to account for course mode
    WF.FAPlusCount[pn] = {0,0}
end
WF.TrackFAPlus = function(pn, judgedata)
    -- Pass player number and params from JudgmentMessage into this
    -- exit under irrelevant conditions
    if not judgedata.TapNoteOffset then return end
    if judgedata.HoldNoteScore then return end
    if judgedata.TapNoteScore == "TapNoteScore_Miss" then return end
    if GAMESTATE:GetPlayerState("PlayerNumber_P"..pn):GetPlayerController() == "PlayerController_Autoplay" then
        return
    end

    local offset = judgedata.TapNoteOffset
    if math.abs(offset) <= 0.010 then
        WF.FAPlusCount[pn][1] = WF.FAPlusCount[pn][1] + 1
        WF.FAPlusCount[pn][2] = WF.FAPlusCount[pn][2] + 1
    elseif math.abs(offset) <= 0.0125 then
        WF.FAPlusCount[pn][2] = WF.FAPlusCount[pn][2] + 1
    end
end


-- ECFA7 exclusive setting -- not putting this in SL.Global.ActiveModifiers because I want it to persist while
-- the game is running, but not save to the machine profile.
-- This setting will be an easy option for the purpose of the tournament to set up the game environment for either
-- a normal FA match or an FA+ match, which will in turn set the default results pane to an ECFA specific one.
WF.ECFAMatch = "None"

-- stuff relating to plain text judgments ([TODO] maybe move this to player options file later)
WF.PlainTextJudgmentFont = "Common Normal"
WF.PlainTextJudgmentBaseZoom = 2
WF.PlainTextJudgmentNames = {
    Waterfall = {
        W1 = "PERFECT!",
        W2 = "EXCELLENT",
        W3 = "GREAT",
        W4 = "CLOSE",
        W5 = "ERROR",
        Miss = "MISS"
    },
    ITG = {
        W1 = "FANTASTIC!",
        W2 = "EXCELLENT",
        W3 = "GREAT",
        W4 = "DECENT",
        W5 = "WAY OFF",
        Miss = "MISS"
    }
}